package com.tandbergtv.watchpoint.studio.external.jpf.validation;

import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.java.plugin.registry.Extension;
import org.java.plugin.registry.IntegrityCheckReport;
import org.java.plugin.registry.Extension.Parameter;
import org.java.plugin.registry.IntegrityCheckReport.ReportItem;
import org.java.plugin.registry.IntegrityCheckReport.Severity;

import com.tandbergtv.watchpoint.studio.external.jpf.PluginParameters;
import com.tandbergtv.watchpoint.studio.external.jpf.model.FailureDescriptor;
import com.tandbergtv.watchpoint.studio.external.jpf.model.ResourceGroupDescriptor;

/**
 * Validator utility class for Resource Group Plugin Validation.
 * 
 * @author rprakash
 *
 */
public class ResourceGroupPluginValidator
{
	private static final Logger logger = Logger.getLogger(ResourceGroupPluginValidator.class);
	
	private ResourceGroupPluginValidator()
	{
	}

	public static FailureDescriptor validateDescriptor(ResourceGroupDescriptor descriptor,
													   boolean alreadyExists)
	{
		logger.debug("Entering method...");
		List<String> errorMessages = new ArrayList<String>();
		
		if(alreadyExists)
			errorMessages.add("Resource Group with name: " +
							   descriptor.getName() +
							   " is already defined.");
		
		String message = null;
		if((message = validateName(descriptor.getName())) != null)
			errorMessages.add(message);
		
		message = null;
		if((message = validateAccessLevel(descriptor.getAccessLevel())) != null)
			errorMessages.add(message);

		message = null;
		if((message = validateInternallyAcquired(descriptor.getIsInternallyAcquired())) != null)
			errorMessages.add(message);
		
		if(errorMessages.size() > 0)
		{
			FailureDescriptor fd = new FailureDescriptor(errorMessages);
			fd.setId(descriptor.getSystemId());
			fd.setExtensionId(descriptor.getExtensionId());
			fd.setName(descriptor.getName());
			return fd;
		}
		else
			return null;
	}
	
	private static String validateInternallyAcquired(Boolean isInternallyAcquired) {
		if(isInternallyAcquired == null)
			return "IsInternallyAcquired cannot be blank or just spaces.";
		else
			return null;
	}

	private static String validateAccessLevel(String accessLevel) {
		if(accessLevel == null)
			return "Access Level cannot be blank or just spaces.";
		else
			return null;
	}

	/*
	 * Validate the Name
	 */
	private static String validateName(String name)
	{
		if (name == null)
		{
			return "Name cannot be blank or just spaces.";
		}
		
		return null;
	}

	
	public static FailureDescriptor validateExtension(Extension extension, IntegrityCheckReport report)
	{
		logger.debug("Entering method...");
		List<String> messages = getFrameworkValidationFailureMessages(extension, report);
		return prepareFailureDescriptor(extension, messages);
	}
	
	private static List<String> getFrameworkValidationFailureMessages(Extension extension, IntegrityCheckReport report) {
		List<String> validationFailureMessages = new ArrayList<String>();
		if (!extension.isValid())
		{
			validationFailureMessages.add("The Plugin Extension has failed framework validation.");

			// Get the list of validation errors
			for (ReportItem item : report.getItems())
			{
				if (!(item.getSeverity() == Severity.ERROR))
					continue;

				String sourceId = (item.getSource() != null) ? item.getSource().getId() : null;
				String pluginId = extension.getDeclaringPluginDescriptor().getId();
				String extensionId = extension.getId();
				if (extensionId.equals(sourceId) || pluginId.equals(sourceId))
				{
					validationFailureMessages.add(item.getMessage());
				}
			}
		}
		return validationFailureMessages;
	}

	/*
	 * Generate the FailureDescriptor that provides information about why the Resource Type
	 * Extension has failed validation.
	 */
	private static FailureDescriptor prepareFailureDescriptor(Extension extension, List<String> messages)
	{
		FailureDescriptor result = null;
		if (messages.size() > 0)
		{
			result = new FailureDescriptor();
			result.setPluginId(extension.getDeclaringPluginDescriptor().getId());
			result.setExtensionId(extension.getId());

			// Get the Id
			Parameter param = extension.getParameter(PluginParameters.SYSTEM_ID);
			if(param != null && param.rawValue().trim().length() > 0)
				result.setId(param.valueAsString());

			// Get the Name
			param = extension.getParameter(PluginParameters.NAME);
			if(param != null && param.rawValue().trim().length() > 0)
				result.setName(param.valueAsString());

			result.setErrorMessages(messages);
			result.setSummaryMessage("The Plugin Extension has failed basic validation.");
		}

		return result;
	}


}
