/*
 * Created on Aug 28, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.service.impl;

import static com.tandbergtv.watchpoint.studio.ui.preferences.PreferenceConstants.DEPLOYMENT_KEY; 
import static com.tandbergtv.watchpoint.studio.application.ApplicationPropertyKeys.WORKFLOW_SYSTEM_ID;

import org.eclipse.jface.preference.IPreferenceStore;

import com.tandbergtv.watchpoint.studio.application.ApplicationProperties;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IMessageUIDGenerator;
import com.tandbergtv.watchpoint.studio.ui.WatchPointStudioPlugin;

/**
 * Message UID Generation Strategy implementation.
 * 
 * @author Vijay Silva
 */
public class MessageUIDGenerator implements IMessageUIDGenerator
{
	private static final String MESSAGE_UID_SEPARATOR = ".";

	private static final String NUMERIC_SET = "0123456789";

	private static final String CHARACTER_SET = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";

	/**
	 * @see com.tandbergtv.watchpoint.studio.service.IMessageUIDGenerator#generateMessageUID(Message,
	 *      ResourceType)
	 */
	public String generateMessageUID(Message message, ResourceType resourceType)
	{
		ApplicationProperties properties = ApplicationProperties.getInstance();
		String wfsSystemId = properties.getProperty(WORKFLOW_SYSTEM_ID);
		
		IPreferenceStore prefs = WatchPointStudioPlugin.getDefault().getPreferenceStore();
		String deploymentKey = prefs.getString(DEPLOYMENT_KEY);
		
		if(deploymentKey == null || deploymentKey.isEmpty()){
			String msg = "The Deployment Key is not set. Set this value on Watchpoint preferences.";
			throw new IllegalArgumentException(msg);
		}

		String sourceId = (message.isIncoming()) ? resourceType.getSystemId() : wfsSystemId;
		String destinationId = (message.isIncoming()) ? wfsSystemId : resourceType.getSystemId();
		String messageId = this.generateMessageID(resourceType.getMessageSequenceNumber());

		String messageUID = sourceId + destinationId + messageId + MESSAGE_UID_SEPARATOR
				+ deploymentKey;

		return messageUID.trim();
	}

	/*
	 * Converts a integer value into the Message ID value
	 */
	private String generateMessageID(int value)
	{
		String result = null;

		// Range 1: [0..99] - Use the Numeric Set
		// Range 2: [100..619] - Use first character from Alphabet Set and second from Numeric Set
		// Range 3: [620..1138] - Use first character from Numeric Set and second from Alphabet Set
		// Range 4: [1139..3843] - Use the Alphabet Set

		int numericSize = NUMERIC_SET.length();
		int alphaSize = CHARACTER_SET.length();

		int numberRange = numericSize * numericSize;
		int alphaRange = alphaSize * alphaSize;
		int mixedRange = numericSize * alphaSize;

		/* Generate the upper limits for each range */
		int range1 = numberRange - 1;
		int range2 = range1 + mixedRange;
		int range3 = range2 + mixedRange;
		int range4 = range3 + alphaRange;

		if (value < 0)
		{
			String msg = "The Message UID's sequence number must be generated "
					+ "from an integer value >= 0, got value: " + value;
			throw new IllegalArgumentException(msg);
		}
		else if (value <= range1)
		{
			int index0 = value % numericSize;
			int index1 = value / numericSize;
			result = NUMERIC_SET.charAt(index1) + "" + NUMERIC_SET.charAt(index0);
		}
		else if (value <= range2)
		{
			value -= range1 + 1;
			int index0 = value % numericSize;
			int index1 = value / numericSize;
			result = CHARACTER_SET.charAt(index1) + "" + NUMERIC_SET.charAt(index0);
		}
		else if (value <= range3)
		{
			value -= range2 + 1;
			int index0 = value % alphaSize;
			int index1 = value / alphaSize;
			result = NUMERIC_SET.charAt(index1) + "" + CHARACTER_SET.charAt(index0);
		}
		else if (value <= range4)
		{
			value -= range3 + 1;
			int index0 = value % alphaSize;
			int index1 = value / alphaSize;
			result = CHARACTER_SET.charAt(index1) + "" + CHARACTER_SET.charAt(index0);
		}
		else
		{
			String msg = "The Message UID's sequence number must be generated "
					+ "from an integer value < " + range4 + ", got value: " + value;
			throw new IllegalArgumentException(msg);
		}

		return result;
	}
}
