package com.tandbergtv.watchpoint.studio.ui.editor;

import java.util.EventObject;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.gef.commands.CommandStackListener;
import org.eclipse.gef.editparts.ZoomManager;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.contexts.IContextActivation;
import org.eclipse.ui.contexts.IContextService;
import org.eclipse.ui.ide.IGotoMarker;
import org.eclipse.wst.sse.ui.StructuredTextEditor;
import org.jbpm.gd.common.editor.ContentProvider;
import org.jbpm.gd.common.editor.Editor;
import org.jbpm.gd.common.editor.GraphPage;
import org.jbpm.gd.common.editor.GraphicalViewer;
import org.jbpm.gd.common.editor.OutlineViewer;
import org.jbpm.gd.common.editor.SelectionSynchronizer;
import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.jpdl.editor.JpdlSelectionSynchronizer;
import org.w3c.dom.Document;

import com.tandbergtv.watchpoint.studio.ui.deployment.JpdlDeploymentEditorPage;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

/**
 * The editor for WatchPoint templates. This class extends JBPM's JpdlEditor and uses WatchPoint
 * templates instead of regular process definitions.
 * 
 * @author Imran Naqvi
 * 
 */
public class WatchPointTemplateEditor extends Editor implements IPartListener2, IGotoMarker {

    private static final String PROCESSIMAGE_JPG = "processimage.jpg";

    private IContextActivation templateEditorContextActivation = null;

    private GraphPage graphPage;

    /**
     * @see org.jbpm.gd.jpdl.editor.JpdlEditor#init(org.eclipse.ui.IEditorSite,
     *      org.eclipse.ui.IEditorInput)
     */
    @Override
    public void init(IEditorSite site, IEditorInput input) throws PartInitException {
        super.init(site, input);

        Utility.setWindow(site.getWorkbenchWindow());
        getActivePage();
        initCommandStackListener();

        checkTemplateProject();
    }

    /**
     * 		Checks if the SuperState is being used in any templates and shows a warning.
     */
    protected void checkTemplateProject() {
        Display display = getSite().getShell().getDisplay();
        display.asyncExec(new Runnable() {
            @Override
            public void run() {
                if (!isTemplateProject()) {
                    String message = "Template does not belong to a Template Project. Some functionalities may not be available. ";
                    MessageDialog.openWarning(Display.getCurrent().getActiveShell(), "Not a Template Project", message);
                }
            }
        });
    }

    public boolean isTemplateProject() {
        boolean isTemplateProject = false;
        IFileEditorInput input = (IFileEditorInput) getEditorInput();
        try {
            isTemplateProject = input.getFile().getProject().hasNature(com.tandbergtv.watchpoint.studio.builder.WatchpointTemplateNature.NATURE_ID);
        } catch (CoreException e) {
            // Couldn't lookup project nature. Assume its not a template project
        }
        return isTemplateProject;
    }

    /**
     * @see org.eclipse.wst.xml.ui.internal.tabletree.XMLMultiPageEditorPart#isSaveOnCloseNeeded()
     */
    @Override
    public boolean isSaveOnCloseNeeded() {
        return isDirty();
    }

    @Override
    protected SelectionSynchronizer createSelectionSynchronizer() {
        return new JpdlSelectionSynchronizer();
    }

    @Override
    protected ContentProvider createContentProvider() {
        return new WPContentProvider();
    }

    @Override
    protected GraphicalViewer createGraphicalViewer() {
        return new GraphicalViewer(this) {
            @Override
            protected void initEditPartFactory() {
                setEditPartFactory(new GraphicalEditPartFactory(getEditor().getEditorInput()));
            }
        };
    }

    @Override
    protected OutlineViewer createOutlineViewer() {
        return null;
    }

    @Override
    protected SemanticElement createMainElement() {
        return getSemanticElementFactory().createById(SemanticElementConstants.TEMPLATE_SEID);
    }

    @Override
    public String getDefaultImageFileName() {
        return PROCESSIMAGE_JPG;
    }

    /*************************************************************/

    /**
     * It is a workaround for recursive Prevented recursive attempt to activate part exception
     * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
     */
    @Override
    public void setFocus() {
        handleTemplateEditorContext();
        this.getGraphicalViewer().getControl().setFocus();
    }

    private void handleTemplateEditorContext(){
        IContextService contextService = (IContextService) getSite().getService(IContextService.class);

        if(templateEditorContextActivation != null){
            contextService.deactivateContext(templateEditorContextActivation);
            templateEditorContextActivation = null;
        }

        if(this.getActivePage() == 0){
            templateEditorContextActivation = contextService.activateContext("com.tandbergtv.watchpoint.template.editor.context");
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Object getAdapter(Class adapter) {
        if (adapter == ZoomManager.class) {
            return getGraphicalViewer().getProperty(ZoomManager.class.toString());
        }

        return super.getAdapter(adapter);
    }

    /**
     * @see org.jbpm.gd.common.editor.Editor#dispose()
     */
    @Override
    public void dispose() {
        super.dispose();

        // Unregister from services
        this.getEditorSite().getWorkbenchWindow().getPartService().removePartListener(this);
    }

    /**
     * Get the Workflow Template associated with this Editor
     * 
     * @return the Workflow Template
     */
    public WorkflowTemplate getProcessDefinition() {
        return (WorkflowTemplate) getRootContainer().getSemanticElement();
    }

    /**
     * @see org.eclipse.ui.part.WorkbenchPart#getPartName()
     */
    @Override
    public String getPartName() {
        if (this.getRootContainer() == null) {
            return super.getPartName();
        }
        return this.getProcessDefinition().getName();
    }

    /**
     * Gets the document element of the xml in the source page
     * 
     * @return The Root Node in the Document contained in the source page.
     */
    public Document getDocumentElement() {
        StructuredTextEditor sourcePage = getSourcePage();
        Document document = (Document) sourcePage.getAdapter(Document.class);

        return document;
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.jbpm.gd.jpdl.ui.editor.JpdlEditor#createPages()
     */
    @Override
    protected void createPages() {
        super.createPages();
        addPage(1, new JpdlDeploymentEditorPage(this), "Export");


        /* register this editor as a listener to editor lifecycle events */
        this.getEditorSite().getWorkbenchWindow().getPartService().addPartListener(this);
    }

    /*
     * (non-Javadoc)
     * 
     * @see org.jbpm.gd.common.editor.Editor#addPage(int, org.eclipse.ui.IEditorPart,
     *      java.lang.String)
     */
    @Override
    protected void addPage(int index, IEditorPart part, String label) {
        /* Do not add page if it is jbpm's deployment page */
        super.addPage(index, part, label);
    }

    @Override
    protected void initSourcePage() {
        super.initSourcePage();
        /*Set source for all transitions*/
        WorkflowTemplate template = getProcessDefinition();
        template.accept(new TransitionSourceSetupVisitor());
    }

    @Override
    protected void initGraphPage() {
        this.graphPage = new WatchPointTemplateGraphPage(this);
        addPage(0, this.graphPage, "Diagram");
        getContentProvider().addNotationInfo(getRootContainer(), getEditorInput());
    }

    @Override
    public GraphPage getGraphPage() {
        return graphPage;
    }

    /*
     * Gets the source page.
     * 
     * @return
     */
    private StructuredTextEditor getSourcePage() {
        StructuredTextEditor sourcePage = null;
        int pageCount = getPageCount();
        for (int i = 0; i < pageCount; i++) {
            if (getEditor(i) instanceof StructuredTextEditor) {
                sourcePage = (StructuredTextEditor) getEditor(i);
            }
        }
        return sourcePage;
    }

    /*
     * Initializes the listener for the command stack changed event.
     */
    private void initCommandStackListener() {
        CommandStackListener commandStackListener = new CommandStackListener() {
            @Override
            public void commandStackChanged(EventObject event) {
            }
        };
        getCommandStack().addCommandStackListener(commandStackListener);
    }

    // ========================================================================
    // ==================== IPartListener2 Methods Start
    // ========================================================================

    /**
     * @see org.eclipse.ui.IPartListener2#partClosed(org.eclipse.ui.IWorkbenchPartReference)
     */
    @Override
    public void partClosed(IWorkbenchPartReference partRef) {
        if (partRef.getPart(false) != this) {
            return;
        }
    }

    /**
     * @see org.eclipse.ui.IPartListener2#partDeactivated(org.eclipse.ui.IWorkbenchPartReference)
     */
    @Override
    public void partDeactivated(IWorkbenchPartReference partRef) {
    }

    /**
     * @see org.eclipse.ui.IPartListener2#partHidden(org.eclipse.ui.IWorkbenchPartReference)
     */
    @Override
    public void partHidden(IWorkbenchPartReference partRef) {
    }

    /**
     * @see org.eclipse.ui.IPartListener2#partInputChanged(org.eclipse.ui.IWorkbenchPartReference)
     */
    @Override
    public void partInputChanged(IWorkbenchPartReference partRef) {
    }

    /**
     * @see org.eclipse.ui.IPartListener2#partOpened(org.eclipse.ui.IWorkbenchPartReference)
     */
    @Override
    public void partOpened(IWorkbenchPartReference partRef) {
    }

    /**
     * @see org.eclipse.ui.IPartListener2#partVisible(org.eclipse.ui.IWorkbenchPartReference)
     */
    @Override
    public void partVisible(IWorkbenchPartReference partRef) {
    }

    /**
     * @see org.eclipse.ui.IPartListener2#partBroughtToTop(org.eclipse.ui.IWorkbenchPartReference)
     */
    @Override
    public void partBroughtToTop(IWorkbenchPartReference partRef) {
    }

    /**
     * @see org.eclipse.ui.IPartListener2#partActivated(org.eclipse.ui.IWorkbenchPartReference)
     */
    @Override
    public void partActivated(IWorkbenchPartReference partRef) {
    }

    @Override
    public void gotoMarker(final IMarker marker) {
        // Forces to open the editor on the "Diagram" page when opening the editor through a Marker.
        super.setActivePage(0);
    }

    // ========================================================================
    // ==================== IPartListener2 Methods End
    // ========================================================================

}
