package com.tandbergtv.watchpoint.studio.ui.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.jbpm.gd.common.model.SemanticElement;

import com.tandbergtv.watchpoint.studio.ui.util.Utility;

public class DataCollectionManager {
    
    private static DataCollectionManager instance;
    private Map<String, String> messageNodeUrlByUid = new HashMap<String, String>();
    private Map<String, String> superstateUrlByName = new HashMap<String, String>();
    private Map<String, String> resourceTypeUrlByName = new HashMap<String, String>();
    private Map<String, ? super SemanticElement> samplesByUrl = new HashMap<String, SemanticElement>();
    private DataCollectionReader reader;

    public synchronized static DataCollectionManager getInstance() {
        if (instance == null) {
            instance = new DataCollectionManager();
        }
        return instance;
    }

    private DataCollectionManager() {
        try {
            IResource resource = ResourcesPlugin.getWorkspace().getRoot();
            IMarker[] markers = resource.findMarkers("org.jbpm.gd.jpdl.CollectedData", true, IResource.DEPTH_ZERO);
            IMarker marker = null;
            if (markers.length > 0) {
                marker = markers[0];

                // fetch info from correspondingMarker
                String info = (String) marker.getAttribute("description");

                // feed info to the DataCollectionManager
                deSerialize(info);
            }
            
            reader = new DataCollectionReader(messageNodeUrlByUid, superstateUrlByName, resourceTypeUrlByName, samplesByUrl);
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    private void storeData() throws CoreException {
        IResource resource = ResourcesPlugin.getWorkspace().getRoot();
        IMarker[] markers = resource.findMarkers("org.jbpm.gd.jpdl.CollectedData", true, IResource.DEPTH_ZERO);
        IMarker marker = null;
        if (markers.length == 0) {
            marker = resource.createMarker("org.jbpm.gd.jpdl.CollectedData");
        } else {
            marker = markers[0];
        }

        marker.setAttribute("description", serialize());
    }

    private String serialize() {
        StringBuilder result = new StringBuilder();
        result.append("messageNodeUrlByUid |");
        for (String key : messageNodeUrlByUid.keySet()) {
            result.append(key).append("=").append(messageNodeUrlByUid.get(key)).append(";");
        }
        result.append("|, superstateUrlByName |");
        for (String key : superstateUrlByName.keySet()) {
            result.append(key).append("=").append(superstateUrlByName.get(key)).append(";");
        }
        result.append("|, resourceTypeUrlByName |");
        for (String key : resourceTypeUrlByName.keySet()) {
            result.append(key).append("=").append(resourceTypeUrlByName.get(key)).append(";");
        }
        result.append("|");
        
        return result.toString();
    }

    /**
     * info format: mapName | name = value; name2 = value2 |, mapName2 |name = value; name2 = value2 |
     * @param data
     */
    private void deSerialize(String data) {
        String[] maps = data.split(",");
        for (String mapChunk : maps) {
            String[] mapChunkStructure = mapChunk.split("\\|");
            String mapName = mapChunkStructure[0];
            if (mapChunkStructure.length > 1) {
                Map<String, String> map = getMap(mapName);
                String propertyName = null;
                String propertyValue = null;
                String[] pairsChunk = mapChunkStructure[1].split(";");
                for (String pairChunk : pairsChunk) {
                    String nameAndValue[] = pairChunk.split("=");
                    propertyName = nameAndValue[0];
                    propertyValue = nameAndValue[1];
                    map.put(propertyName, propertyValue);
                }
            }
        }
    }

    private Map<String, String> getMap(String mapName) {
        if (mapName.trim().equals("messageNodeUrlByUid")) {
            return messageNodeUrlByUid;
        } else if (mapName.trim().equals("superstateUrlByName")) {
            return superstateUrlByName;
        } else if (mapName.trim().equals("resourceTypeUrlByName")) {
            return resourceTypeUrlByName;
        }
        return null;
    }

    public String collectResourceType(String fullPath) {
        try {
            String name = Utility.readResourceTypeName(fullPath);
            if (name != null) {
                String previousValue = resourceTypeUrlByName.put(name, fullPath);
                if (previousValue != null) {
                    samplesByUrl.clear();
                }
                storeData();
            }
            return name;
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    public String collectMessageNode(String fullPath) {
        try {
            String uid = Utility.readMessageNodeUid(fullPath);
            if (uid != null) {
                String previousValue = messageNodeUrlByUid.put(uid, fullPath);
                if (previousValue != null) {
                    samplesByUrl.remove(fullPath);
                }
                storeData();
            }
            return uid;
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    public String collectSuperstate(String fullPath) {
        try {
            String name = Utility.readSuperstateName(fullPath);
            if (name != null) {
                String previousValue = superstateUrlByName.put(name, fullPath);
                if (previousValue != null) {
                    samplesByUrl.remove(fullPath);
                }
                storeData();
            }
            return name;
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    public String removeResourceType(String fullPath) {
        try {
            String name = Utility.readResourceTypeName(fullPath);
            if (name != null) {
                resourceTypeUrlByName.remove(name);
                samplesByUrl.remove(fullPath);
                storeData();
            }
            return name;
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    public String removeMessageNode(String fullPath) {
        try {
            String uid = Utility.readMessageNodeUid(fullPath);
            if (uid != null) {
                messageNodeUrlByUid.remove(uid);
                samplesByUrl.remove(fullPath);
                storeData();
            }
            return uid;
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    public String removeSuperstate(String fullPath) {
        try {
            String name = Utility.readSuperstateName(fullPath);
            if (name != null) {
                superstateUrlByName.remove(name);
                samplesByUrl.remove(fullPath);
                storeData();
            }
            return name;
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    public void removeAll(String projectName) {
        try {
            removeAllMatchingUrlValue(messageNodeUrlByUid, projectName);
            removeAllMatchingUrlValue(superstateUrlByName, projectName);
            removeAllMatchingUrlValue(resourceTypeUrlByName, projectName);
            removeAllMatchingUrlKey(samplesByUrl, projectName);
            storeData();
        } catch (CoreException e) {
            throw new RuntimeException(e);
        }
    }

    private void removeAllMatchingUrlKey(Map<String, ? super SemanticElement> valueByUrl, String projectName) {
        List<String> keysToBeRemoved = new ArrayList<String>();
        for (String url : valueByUrl.keySet()) {
            if (url.startsWith("/" + projectName)) {
                keysToBeRemoved.add(url);
            }
        }
        
        for (String key : keysToBeRemoved) {
            valueByUrl.remove(key);
        }
    }

    private void removeAllMatchingUrlValue(Map<String, String> urlByKey, String projectName) {
        List<String> keysToBeRemoved = new ArrayList<String>();
        for (String key : urlByKey.keySet()) {
            if (urlByKey.get(key).startsWith("/" + projectName)) {
                keysToBeRemoved.add(key);
            }
        }

        for (String key : keysToBeRemoved) {
            urlByKey.remove(key);
        }
    }

    public DataCollectionReader getReader() {
        return reader;
    }
}