/**
 * 
 */
package com.tandbergtv.watchpoint.studio.ui.properties;

import java.io.IOException;
import java.util.HashSet;
import java.util.InvalidPropertiesFormatException;
import java.util.List;
import java.util.Set;

import org.apache.log4j.Logger;
import org.eclipse.draw2d.ColorConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.views.properties.tabbed.AbstractPropertySection;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.NodeElementContainer;

import com.tandbergtv.watchpoint.studio.ui.UIException;
import com.tandbergtv.watchpoint.studio.ui.model.IDueDateElement;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.util.ValidationMessages;
import com.tandbergtv.watchpoint.studio.util.TimerExpression;
import com.tandbergtv.watchpoint.studio.util.TimerExpressionMarshaller;
import com.tandbergtv.watchpoint.studio.validation.IValidationService;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationServiceFactory;

/**
 * Composite UI for showing the due date section.
 * 
 * @author Imran Naqvi
 * 
 */
public class DueDateElementConfigurationComposite implements FocusListener, SelectionListener {

	private static final Logger logger = Logger
			.getLogger(DueDateElementConfigurationComposite.class);

	private TabbedPropertySheetWidgetFactory widgetFactory;

	private Composite parent;

	private CCombo dueDateFirstText;

	private CCombo dueDateOperator;

	private CCombo dueDateSecondText;

	private Label dueDateLabel;

	private IDueDateElement dueDateElement;

	private AbstractPropertySection parentPropertySection;

	private String currentOperator = null;

	private static final String[] ADD_OPERATOR_VARIABLE_FILTER = { "DURATION" };

	private static final String[] MLPY_OPERATOR_VARIABLE_FILTER = { "INT", "STRING" };

	private DueDateElementConfigurationComposite() {
	}

	/**
	 * Creates a DueDateElementConfigurationComposite object in the parent object using the given
	 * widget factory.
	 * 
	 * @param parentSection
	 * @param widgetFactory
	 * @param parent
	 * @return The Composite
	 */
	public static DueDateElementConfigurationComposite create(
			AbstractPropertySection parentSection, TabbedPropertySheetWidgetFactory widgetFactory,
			Composite parent) {
		DueDateElementConfigurationComposite result = new DueDateElementConfigurationComposite();
		result.parentPropertySection = parentSection;
		result.widgetFactory = widgetFactory;
		result.parent = parent;
		result.create();
		return result;
	}

	/**
	 * Sets the due date element and updates controls.
	 * 
	 * @param dueDateElement
	 */
	public void setDueDateElement(IDueDateElement dueDateElement) {
		/* Disable textbox if the node definition's internal node hasnt been set yet */
		if (dueDateElement instanceof NodeDefinition)
			if (((NodeDefinition) dueDateElement).getNode() == null) {
				this.dueDateFirstText.setEnabled(false);
				this.dueDateSecondText.setEnabled(false);
				this.dueDateOperator.setEnabled(false);
			}
			else {
				this.dueDateFirstText.setEnabled(true);
				this.dueDateSecondText.setEnabled(true);
				this.dueDateOperator.setEnabled(true);
			}
		clearControls();
		this.dueDateElement = dueDateElement;
		if (dueDateElement != null) {
			updateControls();
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.events.FocusListener#focusGained(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusGained(FocusEvent e) {
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.events.FocusListener#focusLost(org.eclipse.swt.events.FocusEvent)
	 */
	public void focusLost(FocusEvent e) {
		/* Check if the due date element is present */
		if (this.dueDateElement == null)
			return;

		/* Check if the event is from expected widgets */
		if ((e.widget == dueDateFirstText) || (e.widget == dueDateSecondText)
				|| (e.widget == dueDateOperator)) {
			String firstExpr = dueDateFirstText.getText();
			String operator = dueDateOperator.getText();
			String secondExpr = dueDateSecondText.getText();
			String expression = firstExpr + operator + secondExpr;

			/* Set the expression and validate */
			this.dueDateElement.setDueDate(expression);

			/* Skip validation if the operator is set, but no operand2 is set */
			if (!((operator.length() > 0) && (secondExpr.length() == 0)))
				this.validate();
		}
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent arg0) {
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent arg0) {
		String operator = dueDateOperator.getText();
		if (!operator.equals(this.currentOperator)) {
			setVariables();
			dueDateSecondText.setText("");
			currentOperator = operator;
		}
	}

	/*
	 * Clears the controls.
	 */
	private void clearControls() {
		dueDateFirstText.removeAll();
		dueDateSecondText.removeAll();
		dueDateOperator.select(0);
		this.currentOperator = dueDateOperator.getText();
	}

	/*
	 * Updates the controls.
	 * 
	 */
	private void updateControls() {
		String dueDate = dueDateElement.getDueDate();
		if (dueDate != null) {
			TimerExpression te = TimerExpressionMarshaller.unmarshall(dueDate);
			dueDateFirstText.setText((te.getOperand1() != null) ? te.getOperand1() : "");
			dueDateSecondText.setText((te.getOperand2() != null) ? te.getOperand2() : "");
			dueDateOperator.setText((te.getOperator() != null) ? te.getOperator() : "");
		}
		else {
			dueDateFirstText.setText("");
			dueDateSecondText.setText("");
			dueDateOperator.setText("");
		}

		this.currentOperator = dueDateOperator.getText();
		dueDateFirstText.setItems(getVariables(ADD_OPERATOR_VARIABLE_FILTER));
		this.setVariables();
	}

	/*
	 * Creates the widgets for this composite.
	 * 
	 */
	private void create() {
		Composite composite = widgetFactory.createFlatFormComposite(parent);
		dueDateLabel = new Label(composite, SWT.NONE);
		dueDateLabel.setText("Timeout");
		dueDateLabel.setLayoutData(createDueDateLabelLayoutData());
		dueDateLabel.setBackground(ColorConstants.white);
		dueDateLabel.pack();

		dueDateFirstText = widgetFactory.createCCombo(parent, SWT.DROP_DOWN);
		dueDateFirstText.setLayoutData(createDueDateTextLayoutData(0, 105, 200));
		dueDateFirstText.setItems(new String[] {});
		dueDateFirstText.setEditable(true);
		dueDateFirstText.addFocusListener(this);
		dueDateFirstText.pack();

		dueDateOperator = widgetFactory.createCCombo(parent, SWT.DROP_DOWN);
		dueDateOperator.setLayoutData(createDueDateTextLayoutData(0, 310, 35));
		dueDateOperator.setItems(getOptions());
		dueDateOperator.setEditable(false);
		dueDateOperator.addSelectionListener(this);
		dueDateOperator.addFocusListener(this);
		dueDateOperator.pack();
		this.currentOperator = dueDateOperator.getText();

		dueDateSecondText = widgetFactory.createCCombo(parent, SWT.DROP_DOWN);
		dueDateSecondText.setLayoutData(createDueDateTextLayoutData(0, 370, 200));
		dueDateSecondText.setItems(new String[] {});
		dueDateSecondText.setEditable(true);
		dueDateSecondText.addFocusListener(this);
		dueDateSecondText.pack();
	}

	/*
	 * Creates layout data for the due date label.
	 * 
	 * @return
	 */
	private FormData createDueDateLabelLayoutData() {
		FormData data = new FormData();
		data.left = new FormAttachment(0, -6);
		data.top = new FormAttachment(0, 0);
		return data;
	}

	/*
	 * Creates layout data for the due date text box.
	 * 
	 * @return
	 */
	private FormData createDueDateTextLayoutData(int x, int y, int z) {
		FormData data = new FormData();
		data.left = new FormAttachment(x, y);
		data.top = new FormAttachment(0, 0);
		data.width = z;
		return data;
	}

	/*
	 * Gets the options for the operator
	 */
	private String[] getOptions() {
		String options[] = new String[] { "", "+", "*" };
		return options;
	}

	/*
	 * This method retrieves the defined variables in start state node
	 * 
	 * @return Array of Strings
	 */
	private String[] getVariables(String[] types) {
		IWorkbenchPart part = this.parentPropertySection.getPart();
		List<WPVariable> variables = Utility.getVariables(Utility.getNodeElementContainer(part));
		Set<String> variableNames = new HashSet<String>();
		for (String type : types) {
			for (WPVariable variable : variables) {
				if (type.equals(variable.getType()) && variable.getName() != null) {
					variableNames.add(variable.getName());
				}
			}
		}
		String options[] = variableNames.toArray(new String[variableNames.size()]);

		return options;
	}

	/*
	 * Validates the expression
	 */
	private void validate() {
		ValidationServiceFactory serviceFactory = ValidationServiceFactory.createFactory();
		IValidationService validationService = serviceFactory.createValidationService();
		IWorkbenchPart part = this.parentPropertySection.getPart();
		NodeElementContainer container = Utility.getNodeElementContainer(part);
		List<ValidationMessage> messages = null;
		if (container instanceof WorkflowTemplate) {
			WorkflowTemplate template = (WorkflowTemplate) container;
			messages = validationService.validateDueDate(dueDateElement, template);
		}
		else if (container instanceof NodeDefinition) {
			NodeDefinition nodeDefinition = (NodeDefinition) container;
			messages = validationService.validateDueDate(dueDateElement, nodeDefinition);
		}

		if (messages != null && messages.size() > 0) {
			try {
				String msg = ValidationMessages.getInstance().getMessage(messages.get(0));
				MessageDialog.openError(Display.getCurrent().getActiveShell(),
						"Error in timeout expression", msg);
			}
			catch (InvalidPropertiesFormatException e) {
				logger.error("Error occured while opening the Message Dialog", e);
				throw new UIException("Error occured while opening the Message Dialog");
			}
			catch (IOException e) {
				logger.error("Error occured while opening the Message Dialog", e);
				throw new UIException("Error occured while opening the Message Dialog");
			}
		}
	}

	/*
	 * Sets the variables for operands based on the operator selected
	 */
	private void setVariables() {
		String operator = dueDateOperator.getText();

		boolean enabled = (operator.length() > 0);
		String[] variables = new String[] {};

		if (operator.equals("+")) {
			variables = this.getVariables(ADD_OPERATOR_VARIABLE_FILTER);
		}
		else if (operator.equals("*")) {
			variables = this.getVariables(MLPY_OPERATOR_VARIABLE_FILTER);
		}

		dueDateSecondText.setEnabled(enabled);
		dueDateSecondText.setItems(variables);
	}
}
