package com.tandbergtv.watchpoint.studio.ui.properties;

import org.eclipse.core.resources.IProject;
import org.eclipse.jface.layout.TableColumnLayout;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CellLabelProvider;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TableViewerColumn;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.views.properties.tabbed.TabbedPropertySheetWidgetFactory;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.model.ManualTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;

public class ManualTaskNodeVariablesSectionComposite {

    private TabbedPropertySheetWidgetFactory widgetFactory;
    private Composite parent;
    private Button addButton;
    private Button removeButton;
    private Composite composite;
    private Table table;
    private TableViewer tableViewer;
    private String[] columnTitles = new String[] { "Display Name", "Template Variable", "Required", "Type", "Read", "Write", "Value" };
    private ManualTaskNodeCellModifier cellModifier;
    private TableColumn requiredColumn;
    private TableColumn readColumn;
    private TableColumn writeColumn;

    public ManualTaskNodeVariablesSectionComposite(Composite parent, TabbedPropertySheetWidgetFactory widgetFactory) {
        this.parent = parent;
        this.widgetFactory = widgetFactory;

        createComponents();
    }

	private void createComponents() {
        composite = widgetFactory.createFlatFormComposite(parent);

        createButtons();
        createTable();
        createTableViewer();
    }

    private void createButtons() {
        addButton = widgetFactory.createButton(composite, "+", SWT.PUSH);
        addButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                ManualTaskNode manualTaskNode = (ManualTaskNode) tableViewer.getInput();
                WPVariable variable = manualTaskNode.addEmptyVariable(Utility.getNextAvailableVariableName(table));

                tableViewer.add(variable);
            }
        });
        FormData formData = new FormData();
        formData.top = new FormAttachment(0, 0);
        formData.right = new FormAttachment(100, 0);
        formData.width = 25;
        addButton.setLayoutData(formData);

        removeButton = widgetFactory.createButton(composite, "-", SWT.PUSH);
        formData = new FormData();
        formData.top = new FormAttachment(addButton, 0);
        formData.right = new FormAttachment(100, 0);
        formData.width = 25;
        removeButton.setLayoutData(formData);

        removeButton.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                if (((IStructuredSelection) tableViewer.getSelection()).getFirstElement() != null) {
                    ManualTaskNode manualTaskNode = (ManualTaskNode) tableViewer.getInput();
                    WPVariable variable = (WPVariable) ((IStructuredSelection) tableViewer.getSelection())
                            .getFirstElement();

                    manualTaskNode.removeVariable(variable);

                    tableViewer.remove(variable);
                }
            }
        });
    }

    private void createTable() {
        //table will be created in a exclusive parent in order to use TableColumnLayout
        Composite exclusiveTableParent = widgetFactory.createFlatFormComposite(composite);

        int tableStyle = SWT.SINGLE | SWT.BORDER | SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.HIDE_SELECTION;
        table = widgetFactory.createTable(exclusiveTableParent, tableStyle);
        table.setLinesVisible(true);
        table.setHeaderVisible(true);

        FormData layoutData = new FormData();
        layoutData.left = new FormAttachment(0, 0);
        layoutData.top = new FormAttachment(0, 0);
        layoutData.right = new FormAttachment(addButton, 0);
        layoutData.bottom = new FormAttachment(100, 0);
        layoutData.width = 600;
        exclusiveTableParent.setLayoutData(layoutData);

        final TableColumn displayNameColumn = new TableColumn(table, SWT.LEFT);
        displayNameColumn.setText(columnTitles[0]);

        table.setSortDirection(SWT.DOWN);
        table.setSortColumn(displayNameColumn);
        displayNameColumn.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                table.setSortColumn(displayNameColumn);
                if (table.getSortDirection() == SWT.DOWN) {
                    table.setSortDirection(SWT.UP);
                } else {
                    table.setSortDirection(SWT.DOWN);
                }

                tableViewer.setComparator(new ViewerSorter() {
                    @Override
                    public int compare(Viewer viewer, Object e1, Object e2) {
                        int directionVar = 1;
                        if (table.getSortDirection() == SWT.UP) {
                            directionVar = -1;
                        }

                        Variable v1 = (Variable) e1;
                        Variable v2 = (Variable) e2;

                        if (e1 == null) return directionVar * 1;
                        if (v1.getMappedName() == null) return directionVar * 1;
                        if (e2 == null) return directionVar * -1;
                        if (v2.getMappedName() == null) return directionVar * -1;

                        return directionVar * v1.getMappedName().compareTo(v2.getMappedName());
                    }
                });
            }
        });

        final TableColumn templateVariableColumn = new TableColumn(table, SWT.LEFT);
        templateVariableColumn.setText(columnTitles[1]);

        templateVariableColumn.addSelectionListener(new SelectionAdapter() {
            @Override
            public void widgetSelected(SelectionEvent e) {
                table.setSortColumn(templateVariableColumn);
                if (table.getSortDirection() == SWT.DOWN) {
                    table.setSortDirection(SWT.UP);
                } else {
                    table.setSortDirection(SWT.DOWN);
                }

                tableViewer.setComparator(new ViewerSorter() {
                    @Override
                    public int compare(Viewer viewer, Object e1, Object e2) {
                        int directionVar = 1;
                        if (table.getSortDirection() == SWT.UP) {
                            directionVar = -1;
                        }

                        Variable v1 = (Variable) e1;
                        Variable v2 = (Variable) e2;

                        if (e1 == null) return directionVar * 1;
                        if (v1.getName() == null) return directionVar * 1;
                        if (e2 == null) return directionVar * -1;
                        if (v2.getName() == null) return directionVar * -1;

                        return directionVar * v1.getName().compareTo(v2.getName());
                    }
                });
            }
        });

        requiredColumn = new TableColumn(table, SWT.CENTER);
        requiredColumn.setText(columnTitles[2]);

        TableColumn typeColumn = new TableColumn(table, SWT.LEFT);
        typeColumn.setText(columnTitles[3]);

        readColumn = new TableColumn(table, SWT.CENTER);
        readColumn.setText(columnTitles[4]);
        
        writeColumn = new TableColumn(table, SWT.CENTER);
        writeColumn.setText(columnTitles[5]);

        TableColumn valueColumn = new TableColumn(table, SWT.LEFT);
        valueColumn.setText(columnTitles[6]);

        TableColumnLayout layout = new TableColumnLayout();
        exclusiveTableParent.setLayout(layout);
        layout.setColumnData(displayNameColumn, new ColumnWeightData(40));
        layout.setColumnData(templateVariableColumn, new ColumnWeightData(40));
        layout.setColumnData(requiredColumn, new ColumnWeightData(15));
        layout.setColumnData(typeColumn, new ColumnWeightData(20));
        layout.setColumnData(readColumn, new ColumnWeightData(15));
        layout.setColumnData(writeColumn, new ColumnWeightData(15));
        layout.setColumnData(valueColumn, new ColumnWeightData(45));
    }

    private void createTableViewer() {
        tableViewer = new TableViewer(table);

        tableViewer.setUseHashlookup(true);
        tableViewer.setColumnProperties(columnTitles);
        tableViewer.setContentProvider(new ManualTaskNodeContentProvider());
        tableViewer.setLabelProvider(new ManualTaskNodeLabelProvider(requiredColumn, readColumn, writeColumn));

        CellEditor[] editors = new CellEditor[columnTitles.length];
        editors[0] = new TextCellEditor(table, SWT.BORDER);
        editors[1] = new TextCellEditor(table, SWT.BORDER);
        editors[2] = new CheckboxCellEditor(table);
        editors[3] = new ComboBoxCellEditor(table, Utility.toVariableTypeName(Utility.getWPDataTypes()), SWT.BORDER);
        editors[4] = new CheckboxCellEditor(table);
        editors[5] = new CheckboxCellEditor(table);
        editors[6] = new TextCellEditor(table);

        // Assign the cell editors to the viewer 
        tableViewer.setCellEditors(editors);

        cellModifier = new ManualTaskNodeCellModifier(tableViewer, columnTitles);

        // Set the cell modifier for the viewer
        tableViewer.setCellModifier(cellModifier);

        TableViewerColumn viewerColumn = new TableViewerColumn(tableViewer, requiredColumn);
        viewerColumn.setLabelProvider((CellLabelProvider) tableViewer.getLabelProvider());
        viewerColumn = new TableViewerColumn(tableViewer, readColumn);
        viewerColumn.setLabelProvider((CellLabelProvider) tableViewer.getLabelProvider());
        viewerColumn = new TableViewerColumn(tableViewer, writeColumn);
        viewerColumn.setLabelProvider((CellLabelProvider) tableViewer.getLabelProvider());

        table.pack();
    }

    public void setInput(ManualTaskNode manualTaskNode, IProject projectFromEditor) {
        tableViewer.setInput(manualTaskNode);
   }
}