/**
 * UpdateSuperStateCommand.java
 * Created Apr 24, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.resource;

import static javax.xml.xpath.XPathConstants.NODESET;

import java.util.Iterator;

import javax.xml.transform.TransformerException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathException;
import javax.xml.xpath.XPathFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.SynchronizationException;
import com.tandbergtv.watchpoint.studio.ui.sync.util.ISynchronizationContext;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.FileUtil;

/**
 * Updates a superstate node. Modifies the DOM model to include the correct IDs and fixes relationships
 * with messages and child nodes.
 * 
 * @author Sahil Verma
 */
public class UpdateSuperStateCommand extends UpdateNodeCommand {

	/**
	 * Creates a command
	 * 
	 * @param name
	 * @param diff
	 * @param context
	 */
	public UpdateSuperStateCommand(String name, IDiff diff, ISynchronizationContext context) {
		super(name, diff, context);
	}

	/**
	 * {@inheritDoc}
	 */
	protected void updateRelationships() {
		addChildren();
		removeChildren();
	}
	
	/**
	 * Copy children from the new superstate to the existing one
	 */
	protected void addChildren() {
		for (NodeDefinitionDTO child : this.model.getChildren()) {
			if (!(this.localModel.getChildren().contains(child))) {
				localModel.addChild(child);
				
				Message message = child.getMessages().iterator().next();
				
				localModel.addMessage(message);
			}
		}
	}

	/**
	 * Removes relationship with all child nodes that have been removed in the new version
	 */
	protected void removeChildren() {
		Iterator<NodeDefinitionDTO> i = this.localModel.getChildren().iterator();
		
		while (i.hasNext()) {
			NodeDefinitionDTO child = i.next();
			
			/* If the UI model does not contain the node */
			if (element.getNodeDefinitionByDefinitionName(child.getName()) == null) {
				i.remove();
				
				/* Also remove the message that is not used anymore */
				Message message = child.getMessages().iterator().next();
				
				this.localModel.removeMessage(message);
			}
		}
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	protected void updateNodeDefinitionXml() {
		/* FIXME Might be easier to convert element to DOM because it is complete at this point */
		try {
			Document document = Utility.getDocumentFromString(model.getXml());
			Element e = document.getDocumentElement();

			e.setAttribute("resourceType", String.valueOf(model.getResourceType().getId()));

			XPath xpath = XPathFactory.newInstance().newXPath();
			NodeList nodelist = (NodeList) xpath.evaluate("//super-state/nodeDefinition", e, NODESET);

			/* Set the persisted id and resourceType id of each child node */
			for (int i = 0; i < nodelist.getLength(); i++) {
				Element dom = (Element) nodelist.item(i);
				String name = dom.getAttribute("definitionName");

				dom.setAttribute("resourceType", String.valueOf(model.getResourceType().getId()));

				for (NodeDefinitionDTO child : model.getChildren()) {
					if (child.getName().equals(name)) {
						dom.setAttribute("id", String.valueOf(child.getId()));
						break;
					}
				}
			}

			String xml = FileUtil.convertDocument(document);

			model.setXml(xml);
		} catch (XPathException e) {
			throw new SynchronizationException("Failed to convert node XML", e);
		} catch (TransformerException e) {
			throw new SynchronizationException("Failed to convert node XML", e);
		}
	}
}
