package com.tandbergtv.watchpoint.studio.ui.view.resourcetype.actions;

import static com.tandbergtv.watchpoint.studio.service.ServiceFactory.createFactory;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.ADAPTOR_FOLDER_PATH;
import static org.eclipse.jface.dialogs.MessageDialog.openError;
import static org.eclipse.swt.widgets.Display.getCurrent;

import java.io.ByteArrayInputStream;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.window.Window;
import org.jbpm.gd.common.xml.XmlAdapter;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.ResourceTypeMessageRenameDialog;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

public class RenameMessageAction extends AbstractNodeDefinitionAction {
	
	private NodeDefinitionDTO dto;
	private XmlAdapter adapter;
	private IFile templateFile;
	private String currentNodeDefinitionName;
	private String newName;
	
	@Override
	public void run(IAction action) {
		if (dto == null)
			return;
		
		NodeDefinition node = null;
		try {
			node = SemanticElementUtil.createNodeDefinition(dto.getXml());
			node.setId(dto.getId());
        } catch (Exception e) {
			handleException("Error creating message node from xml: " + dto.getName(), e);
			return;
		}
		
		/* get the references for the node if the node is being used by templates or superstates*/
		this.nodeReferences = checkNodeInUse(node);
		/* build the rename dialog */
		ResourceTypeMessageRenameDialog dialog = buildDialog(node);
		/* set the current node defintion name */
		currentNodeDefinitionName = dto.getName();
		
		/* opens the dialog */
		if (dialog.open() != Window.OK)
			return;
		/* check the dialog parameters */
		if (!checkDialogParameters(dialog))
			return;
		
		newName = dialog.getMessageName();
		
		/* Rename the node definition in the resource type and also in superstates which contains
		 * this node definition*/
		renameNodeDefinition(dialog.getMessageName(), node);
		
		/* If the user choose to update the references, refactor the affected templates */
		if(dialog.getUpdateReferences()){
			try {
				new TemplateNodesRefactoring().refactorMessageNodes(dialog.getAffectedMessages(), dto, currentNodeDefinitionName);
			} catch (InvocationTargetException e) {
				handleException("Error refactoring the message nodes", e);
			} catch (InterruptedException e) {
				handleException("Error refactoring the message nodes", e);
			}
		}
	}
	
	@Override
	protected void updateNodeInSuperstate(NodeDefinition nodeSuperstate) {
		nodeSuperstate.setDefinitionName(newName);
	}
	
	private void renameNodeDefinition(String newName, NodeDefinition node){
		
		INodeDefinitionService service = ServiceFactory.createFactory().createNodeDefinitionService();
		// Remove the current node definition --------
		service.deleteNodeDefinitionInFileSystem(dto, true);
		ResourceType resourceType = dto.getResourceType();
		resourceType.removeNode(dto);
		Message message = resourceType.getMessage(node.getUid());
		resourceType.removeMessage(message);
		
		// Delete adaptor definition for the message
		IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(resourceType.determineNameForProject());
		
		if (project.exists()) {
			try {
				project.getFolder(ADAPTOR_FOLDER_PATH + message.getUid()).delete(true, true, null);
			} catch (CoreException e) {
				handleException("Message deleted but unable to delete message folder from the resource type project.", e);
			}
		}
		// --------
		
		// Set the node definition new name 
		XmlAdapter adapter = Utility.getAdapterFromString(dto.getXml(), node.getFactory());
		adapter.initialize(node);
		
		node.setName(newName);
		dto.setName(newName);
		message.setName(newName);
		
		
		/* Check if the node definition is being used in a superstate. If true, updates the
		   superstate */
		for (IWatchPointDTO referenceDTO : this.nodeReferences) {
			if(referenceDTO instanceof NodeDefinitionDTO){
				NodeDefinitionDTO nodeDTO = (NodeDefinitionDTO) referenceDTO;
				if(nodeDTO.getType().equals(NodeDefinitionType.SuperState)){
					updateSuperstateNodeDefinition(nodeDTO, node, currentNodeDefinitionName);
				}
			}
		}
		
		
		// Create the new nodedefinition
		String xml = Utility.getXMLFromDocument(adapter.getNode().getOwnerDocument());
		dto.setXml(xml);
		
		service.createNodeDefinitionInFileSystem(dto);
		resourceType.addMessage(message);
		
		IResourceTypeService rtService = createFactory().createResourceTypeService();
		rtService.updateResourceTypeInFileSystem(resourceType);
		
		ResourceType updated = rtService.getResourceTypeFromFileSystem(resourceType.getProjectName());
		if (updated == null) {
			Utility.reportError("Error updating resource type", null);
		} 
		
	}
	
	
	/**
	 * 	    Saves the Template on FileSystem.
	 * 
	 */
	protected void saveTemplate() throws CoreException {
		String xml = Utility.getXMLFromDocument(adapter.getNode().getOwnerDocument());
		
		ByteArrayInputStream input = new ByteArrayInputStream(xml.getBytes());
		templateFile.setContents(input, true, false, null);
	}
	
	protected ResourceTypeMessageRenameDialog buildDialog(NodeDefinition node) {
		ResourceTypeMessageRenameDialog dialog = 
			new ResourceTypeMessageRenameDialog(getCurrent().getActiveShell());
		
		dialog.initReferences( buildReferencesMap(dto) );
		dialog.setMessageName(dto.getName());
		
		dialog.setBlockOnOpen(true);
		
		return dialog;
	}
	
		
	protected boolean checkDialogParameters(ResourceTypeMessageRenameDialog dialog) {
		String name = dialog.getMessageName();
		
		if (name == null || name.length() == 0) {
			openError(getCurrent().getActiveShell(), "Error", "Variable name can not be empty.");
			return false;
		}
		
		if (name.contains(":")) {
			openError(getCurrent().getActiveShell(), "Error", "Variable name can not contain ':' character.");
			return false;
		}
		
		return true;
	}
	
	@Override
	public void selectionChanged(IAction action, ISelection selection) {
		if (!selection.isEmpty()) {
			Object element = ((TreeSelection) selection).getFirstElement();
			if (element instanceof NodeDefinitionDTO) {
				this.dto = (NodeDefinitionDTO) element;
			}
		}
	}


}
