/**
* ImportWizardDestinationProjectPage.java
 * Created Apr 27, 2011
 */
package com.tandbergtv.watchpoint.studio.ui.wizard.imports;

import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.jbpm.gd.jpdl.Logger;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.external.imports.ImportDataProvider;
import com.tandbergtv.watchpoint.studio.ui.wizard.AbstractWizardPage;

/**
 * Prompts for project/folder where items will be imported
 * 
 * @author Francisco Bento da Silva Neto
 */
public class ImportWizardDestinationProjectPage extends AbstractWizardPage {
	
	private static final int MAXENTRIES = 5;
	
	private Text projectNameField;
	
	private Button browseButton;
	
	private Label label;
	
	private String containerName;
	
	private Map<Class<? extends IWatchPointDTO>, ImportDataProvider> map;
	
	private Class<? extends IWatchPointDTO> clazz;
	
	private Collection<? extends IWatchPointDTO> models;
	
	private String[] requiredProjectNatures;
	
	/**
	 * Creates a page
	 * 
	 * @param pageName
	 */
	public ImportWizardDestinationProjectPage(String pageName) {
		super(pageName);
		setTitle("Target location");
		setMessage("Workspace location where the items will be imported");
		this.map = new HashMap<Class<? extends IWatchPointDTO>, ImportDataProvider>();
	}

	/**
	 * @param pageName
	 * @param clazz
	 */
	public ImportWizardDestinationProjectPage(String pageName, Class<? extends IWatchPointDTO> clazz) {
		this(pageName);
		this.clazz = clazz;
	}

	public void addProvider(Class<? extends IWatchPointDTO> clazz, ImportDataProvider provider) {
		this.map.put(clazz, provider);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public boolean isPageComplete() {
		return getBaseDirectory() != null && getBaseDirectory().trim().length() > 0;
	}
	
	public Collection<? extends IWatchPointDTO> loadModel(String location) {
		fetchItems(location);

		return models;
	}
	
	/**
	 * {@inheritDoc}
	 */
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite container = createClientArea(parent);
		
		Dialog.applyDialogFont(container);
		initialize();
		setControl(container);
	}
	
	/**
	 * @param parent
	 * @return
	 */
	protected Composite createClientArea(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		FormLayout layout = new FormLayout();
		
		layout.marginWidth = 10;
		layout.marginHeight = 50;
		
		composite.setLayout(layout);
		
		FormData formData = new FormData();
		
		this.label = new Label(composite, SWT.NONE);
		this.label.setText("Directory:");
		formData.left = new FormAttachment(0);
		this.label.setLayoutData(formData);
		
		this.browseButton = new Button(composite, SWT.PUSH);
		this.browseButton.setText("Browse");
		this.browseButton.setEnabled(true);
		
		formData = new FormData();
		formData.top = new FormAttachment(0);
		formData.right = new FormAttachment(100, -10);
		
		this.browseButton.setLayoutData(formData);

		this.projectNameField = new Text(composite, SWT.BORDER);
		
		formData = new FormData();
		formData.top = new FormAttachment(0);
		formData.left = new FormAttachment(this.label, 5);
		formData.right = new FormAttachment(this.browseButton, -10);
		
		this.projectNameField.setLayoutData(formData);
		
		return composite;
	}

	protected void initialize() {
		hookListeners();
	}
	
	/**
	 * Attach listeners to widgets on the page
	 */
	protected void hookListeners() {
		this.browseButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				chooseContainer();
			}
		});
		
		this.projectNameField.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				validate();
			}
		});
	}
	
	private void chooseContainer() {
		WorkbenchContentProvider provider= new WorkbenchContentProvider();
		ILabelProvider labelProvider= new WorkbenchLabelProvider(); 
		ElementTreeSelectionDialog dialog= new ElementTreeSelectionDialog(getShell(), labelProvider, provider);
		dialog.setTitle("Folder Selection");
		dialog.setMessage("Choose a folder");
		dialog.setInput(ResourcesPlugin.getWorkspace());
		dialog.addFilter(createViewerFilter());
		dialog.open();
		initContainerName((IContainer)dialog.getFirstResult());
		projectNameField.setText(containerName);
	}
	
	private void initContainerName(IContainer elem) {
		containerName = (elem == null) ? "" : elem.getFullPath().makeRelative().toString(); 
	}
	
	private ViewerFilter createViewerFilter() {
		ViewerFilter filter= new ViewerFilter() {
			public boolean select(Viewer viewer, Object parent, Object element) {
				boolean filter = false;
				
				IProject project = null;
				if (IFolder.class.isInstance(element)) {
					IFolder folder = (IFolder) element;
					project = folder.getProject();
				} else if (IProject.class.isInstance(element)){
					project = (IProject) element;
				}
				if (project != null) {
					try {
						if (requiredProjectNatures == null || requiredProjectNatures.length == 0) {
							// No filter per-nature
							filter = true;
						} else {
							// Will only show projects that have the defined natures
							for (String projectNature : requiredProjectNatures) {
								if (project.hasNature(projectNature)) {
									filter = true;
									break;
								}
							}
						}
					} catch (CoreException e) {}
				}
				return filter;
			}
		};
		return filter;
	}
	
	/**
	 * @return a chosen path from the directory dialog invoked from browse button
	 */
	protected IPath chooseDirectory() {
		DirectoryDialog dialog = new DirectoryDialog(getShell());
		
		dialog.setFilterPath(projectNameField.getText());
		dialog.setText("Choose import directory");
		dialog.setMessage("Select a directory containing items to import");
		
		String res = dialog.open();
		
		if (res != null)
			return new Path(res);
		
		return null;
	}
	
	protected void fetchItems(final String directory) {
		IRunnableWithProgress op = new IRunnableWithProgress() {
			public void run(IProgressMonitor monitor) {
				monitor.beginTask("Calculating items to import...", IProgressMonitor.UNKNOWN);
				ImportDataProvider provider = map.get(clazz);
				
				provider.setImportLocation(directory);
				models = provider.getItems();
				monitor.done();
			}
		};
		
		try {
			getContainer().run(true, false, op);
		} catch (InvocationTargetException e) {
			Logger.logError(e.getCause());
			/* FIXME Message dialog */
		} catch (InterruptedException e) {
		}
	}
	
	/**
	 * Returns the directory from which we're going to import plugins
	 * 
	 * @return
	 */
	public  String getBaseDirectory() {
		return this.projectNameField.getText();
	}
	
	/**
	 * Validates the drop location
	 */
	protected void validate() {
		if (browseButton.getSelection()) {
			IPath curr = new Path(projectNameField.getText());
			
			if (curr.segmentCount() == 0 && curr.getDevice() == null) {
				setErrorMessage("Missing directory");
				setPageComplete(false);
				return;
			}
			
			if (!Path.ROOT.isValidPath(projectNameField.getText())) {
				setErrorMessage("Invalid folder");
				setPageComplete(false);
				return;
			}

			if (!curr.toFile().isDirectory()) {
				setErrorMessage("Not a directory");
				setPageComplete(false);
				return;
			}
		}
		
		setErrorMessage(null);
		setPageComplete(true);
	}
	
	/**
	 * {@inheritDoc}
	 */
	@Override
	public void saveSettings(IDialogSettings settings) {
	}

	/**
	 * Saves most recent entries to the settings plus current
	 * 
	 * @param settings
	 * @param key
	 * @param combo
	 */
	protected void saveCombo(IDialogSettings settings, String key, Combo combo) {
		if (combo.getText().trim().length() > 0) {
			settings.put(key + String.valueOf(0), combo.getText().trim());
			String [] items = combo.getItems();
			int count = Math.min(items.length, MAXENTRIES);
			
			for (int i = 0; i < count; i++)
				settings.put(key + String.valueOf(i + 1), items[i].trim());
		}
	}

	public String[] getRequiredProjectNatures() {
		return requiredProjectNatures;
	}

	public void setRequiredProjectNatures(String... requiredProjectNatures) {
		this.requiredProjectNatures = requiredProjectNatures;
	}
	
}
