/**
 * TemplateImportService.java
 * Created Feb 22, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.wizard.template;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.Enumeration;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;

import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.FileUtil;

/**
 * Imports a template into the current workspace
 * 
 * @author Sahil Verma
 * @author Francisco Bento da Silva Neto
 */
public class TemplateImportService {

	private static final String CLASSES_DIR = "classes/";
	private static final String CLASSES_JAR_FILE_NAME = "classes.jar";
	
	private static final String CLASSES_JAR_FOLDER = "lib/";

	public TemplateImportService() {
		super();
	}

	/**
	 * 		Imports the specified items into the specified folder.
	 * 
	 * @param templates
	 * @param destinationFolder
	 */
	public void importItems(Collection<WorkflowTemplateDTO> templates, IFolder destinationFolder) {
		for (WorkflowTemplateDTO template : templates) {
			if (template.getPath().endsWith(WorkflowTemplateDTO.DEFAULT_TEMPLATE_PACKAGE_EXTENSION)) {
				// Its a PAR template file
				importTemplatePackage(template, destinationFolder);
			} else {
				importTemplateFiles(template, destinationFolder);				
			}
		}
	}
	
	/**
	 * 		Import a template package (zip).
	 * 
	 * @param template
	 * @param baseDestinationFolder
	 */
	protected void importTemplatePackage(WorkflowTemplateDTO template, IFolder baseDestinationFolder) {
		ZipFile packageFile = null;
		try {
			IFolder templateRootFolder = baseDestinationFolder.getFolder(template.getName());
			if (!templateRootFolder.exists()) {
				templateRootFolder.create(true, true, null);
			}
			
			packageFile = new ZipFile(new File(template.getPath()));

			ByteArrayOutputStream baos = new ByteArrayOutputStream();
			ZipOutputStream jarFile = new ZipOutputStream(baos);
			boolean createClassesJarFile = false;
			
			Enumeration<? extends ZipEntry> entries = packageFile.entries();
			while (entries.hasMoreElements()) {
				ZipEntry entry = entries.nextElement();
				if (entry.getName().startsWith(CLASSES_DIR) && !entry.isDirectory()) {
					String newEntryName = entry.getName().substring(CLASSES_DIR.length());
					
					ZipEntry newEntry = new ZipEntry(newEntryName);
					jarFile.putNextEntry(newEntry);

					byte[] b = FileUtil.readFile(packageFile.getInputStream(entry));
					jarFile.write(b);
					createClassesJarFile = true;
				} else {
					InputStream stream = packageFile.getInputStream(entry);
					IFile file = templateRootFolder.getFile(entry.getName());
					if (file.exists()) {
						file.setContents(stream, true, false, null);
					} else {
						file.create(stream, true, null);
					}
				}
			}
			if (createClassesJarFile) {
				jarFile.close();
				InputStream in = new ByteArrayInputStream(baos.toByteArray());
				
				IFolder classesJarFolder = templateRootFolder.getProject().getFolder(new Path(CLASSES_JAR_FOLDER));
				if (!classesJarFolder.exists()) {
					classesJarFolder.create(true, true, null);
				} 

				IFile classesJar = classesJarFolder.getFile(template.getName() + "_" + CLASSES_JAR_FILE_NAME);
				if (!classesJar.exists()) {
					classesJar.create(in, true, null);
				} else {
					classesJar.setContents(in, true, true, null);
				}
				// adds the jar to the classpath
				IClasspathEntry newClasspathEntry = JavaCore.newLibraryEntry(classesJar.getFullPath(), null, null);
				IJavaProject javaProject = JavaCore.create(classesJar.getProject());
				IClasspathEntry[] classpathEntries = javaProject.getRawClasspath();
				
				boolean existsInClassPath = false;
				for (IClasspathEntry entry : classpathEntries) {
					if (entry.getPath().equals(newClasspathEntry.getPath())) {
						existsInClassPath = true;
					}
				}
				
				if (!existsInClassPath) {
					IClasspathEntry[] newEntries = new IClasspathEntry[classpathEntries.length + 1];
					System.arraycopy(classpathEntries, 0, newEntries, 0, classpathEntries.length);
					newEntries[newEntries.length - 1] = newClasspathEntry;
					
					javaProject.setRawClasspath(newEntries, null);
				}
			}
		} catch (Exception e) {
			Utility.reportError("Error import Template: " + template.getName(), e);
		} finally {
			if (packageFile != null) {
				try {
					packageFile.close();
				} catch (IOException e) { }
			}
		}
	}

	/**
	 * 		Imports the template files.
	 * 
	 * @param template
	 * @param baseDestinationFolder
	 */
	protected void importTemplateFiles(WorkflowTemplateDTO template, IFolder baseDestinationFolder) {
		/* It is not a par, so we have two different cases:
		 * 	Old-style, where templates were named:
		 *  	<Template Name>.jpdl
		 *  	<Template Name>.gpd
		 *  	<Template Name>.xml
		 *  New-style, where templates were named:
         *      <Template Name>.jpdl
		 *  	processdefinition.gpd
		 *  	processdefinition.xml
		 *      processdefinition.jpg
		 */
		
		String path = template.getPath();
		File sourceFolder = new File(path);
		if (!sourceFolder.isDirectory()) {
			sourceFolder = sourceFolder.getParentFile();
		}
		
		String templateName = path.substring(0,path.lastIndexOf("."));
		File gpdFile = new File(templateName + WorkflowTemplateDTO.GPD_FILE_EXTENSION);
		if (gpdFile.exists()) {
			// Its a old-style template, because there's another file in folder called <Template Name>.gpd.
			// In this case, import just the 3 files.
			IFolder templateRootFolder = baseDestinationFolder.getFolder(template.getName());
			try {
				File templateJPDL = new File(path);
				FileUtil.importFilesystemFolderToWorkspace(templateJPDL, templateRootFolder, false);
				FileUtil.importFilesystemFolderToWorkspace(gpdFile, templateRootFolder, false);
				
				// there is no need to import XML as its gonna be re-generated by the builder
				// rename the <Template Name>.gpd to processdefinition.gpd
				IFile newGPD = templateRootFolder.getFile(gpdFile.getName());
				IFile previousGPD = templateRootFolder.getFile(WorkflowTemplateDTO.GPD_FILE_NAME);
				// deletes the previous gpd file if it exists 
				if (previousGPD.exists()) {
					previousGPD.delete(true, null);
				}
				newGPD.move(previousGPD.getFullPath(), true, null);	
			} catch (Exception e) {
				Utility.reportError("Error import Template: " + template.getName(), e);
			}
		} else {
			// <Template Name>.gpd doesnt exists, so lets assume its a new-style template.
			// Import everything in the folder.
			try {
				IFolder templateRootFolder = baseDestinationFolder.getFolder(template.getName());
				FileUtil.importFilesystemFolderToWorkspace(sourceFolder, templateRootFolder, true);
			} catch (Exception e) {
				Utility.reportError("Error import Template: " + template.getName(), e);
			}
		} 
	}
	
}