package com.tandbergtv.watchpoint.studio.usermgmt;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;

/**
 * The User defines the a set of data that will help uniquely identify each user of the WatchPoint
 * Plug-in. The User object maintains information about the userName maintained by the OS, the IP
 * Address of the machine on which the plug-in is executed, a time stamp at which the user was first
 * created, and a random number generated. These fields look to uniquely identify each user of the
 * WatchPoint Plug-in.
 * 
 * @author DReddy
 */
public class User implements IWatchPointDTO {

	// Serialization Version Number
	private static final long serialVersionUID = -727719057351564588L;

	private String userName;

	private String ipAddress;

	private long timeStamp;

	private int randomNumber;

	/**
	 * @param userName
	 *            The User Name
	 * @param ipAddress
	 *            The IP Address of the machine the user is operating on
	 * @param timeStamp
	 *            The Time Stamp at which the user was created / generated
	 * @param randomNumber
	 *            A Random integer value
	 */
	public User(String userName, String ipAddress, long timeStamp, int randomNumber) {
		super();
		this.userName = userName;
		this.ipAddress = ipAddress;
		this.timeStamp = timeStamp;
		this.randomNumber = randomNumber;
	}

	/**
	 * Gets the User Name
	 * 
	 * @return the userName
	 */
	public String getUserName() {
		return userName;
	}

	/**
	 * Gets the IP Address (as a String value)
	 * 
	 * @return the ipAddress
	 */
	public String getIpAddress() {
		return ipAddress;
	}

	/**
	 * Gets the Time Stamp (in ms) at which the User was first generated
	 * 
	 * @return the timeStamp
	 */
	public long getTimeStamp() {
		return timeStamp;
	}

	/**
	 * A random number used to avoid collisions in generated User objects
	 * 
	 * @return the Random Number
	 */
	public int getRandomNumber() {
		return randomNumber;
	}

	/**
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode() {
		final int prime = 13;
		int result = 1;
		result = prime * result + ((ipAddress == null) ? 0 : ipAddress.hashCode());
		result = prime * result + randomNumber;
		result = prime * result + (int) (timeStamp ^ (timeStamp >>> 32));
		result = prime * result + ((userName == null) ? 0 : userName.hashCode());
		return result;
	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj)
			return true;
		if (obj == null)
			return false;
		if (getClass() != obj.getClass())
			return false;
		final User other = (User) obj;
		if (ipAddress == null) {
			if (other.ipAddress != null)
				return false;
		} else if (!ipAddress.equals(other.ipAddress))
			return false;
		if (randomNumber != other.randomNumber)
			return false;
		if (timeStamp != other.timeStamp)
			return false;
		if (userName == null) {
			if (other.userName != null)
				return false;
		} else if (!userName.equals(other.userName))
			return false;
		return true;
	}

	/**
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString() {
		StringBuffer sb = new StringBuffer(userName);
		sb.append(":");
		sb.append(ipAddress);
		sb.append(":");
		sb.append(timeStamp);
		sb.append(":");
		sb.append(randomNumber);
		return sb.toString();
	}
}
