/*
 * Created on Jul 20, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.graph;

import java.util.HashSet;
import java.util.Set;

import org.jbpm.gd.jpdl.model.Fork;
import org.jbpm.gd.jpdl.model.Join;

import edu.uci.ics.jung.graph.Edge;
import edu.uci.ics.jung.graph.Vertex;
import edu.uci.ics.jung.utils.UserData;

/**
 * The Fork Path Labeler labels every Fork Node with a nest level number indicating the nested level
 * of the fork node, where level 1 indicates that the Fork Node is not nested. The labeler also
 * labels every Fork node and outgoing Fork transition with the join vertices label and the
 * reachable vertex label. The join vertices label maintains a set of join vertices that the
 * transition eventually reaches. The reachable vertex label stores a set of vertices reachable from
 * the fork transition to the Join node without including the Fork or Join node.
 * 
 * @author Vijay Silva
 */
public class ForkPathLabeler
{
	/**
	 * Default Label used for decorating Fork Nodes with the nest level integer value
	 */
	public static final String DEFAULT_FORK_NEST_LEVEL_KEY = "fork.NestLevel";

	/**
	 * Default Label used for decorating Fork Nodes and Transitions with the paired Join Vertices.
	 */
	public static final String DEFAULT_JOIN_VERTICES_KEY = "fork.JoinVertices";

	/**
	 * Default Label used for decorating Fork Nodes and Transitions with all reachable nodes upto
	 * but excluding the paired Join nodes.
	 */
	public static final String DEFAULT_REACHABLE_VERTICES_KEY = "fork.reachableVertices";

	/**
	 * Default Label used for decorating Join Nodes with all the Fork Nodes paired with it.
	 */
	public static final String DEFAULT_FORK_VERTICES_KEY = "join.ForkVertices";

	private String forkNestLevelKey;

	private String joinVerticesKey;

	private String reachableVerticesKey;

	private String forkVerticesKey;

	private WatchPointDecorator<String, Integer> nestLevelDecorator;

	private WatchPointDecorator<String, Set<Vertex>> joinVerticesDecorator;

	private WatchPointDecorator<String, Set<Vertex>> reachableVerticesDecorator;

	private WatchPointDecorator<String, Set<Vertex>> forkVerticesDecorator;

	private Set<Vertex> forkVertices;

	private Set<Vertex> joinVertices;

	private IWatchPointGraph graph;

	/**
	 * Constructor
	 */
	public ForkPathLabeler()
	{
		this(null, null, null, null);
	}

	/**
	 * Constructor. To use any of the default key values, use null as the key input value.
	 * 
	 * @param nestKey
	 *            The Key to use for labeling fork vertices with the nest level.
	 * @param joinKey
	 *            The Key to use for labeling the fork vertices and transitions with the set of join
	 *            vertices
	 * @param reachableVertexKey
	 *            The Key to use for labeling the fork vertices and transitions with the set of
	 *            reachable vertices
	 * @param forkKey
	 *            The Key to use for labeling the join vertices with the set of paired fork
	 *            vertices.
	 */
	public ForkPathLabeler(String nestKey, String joinKey, String reachableVertexKey, String forkKey)
	{
		/* Set the keys */
		this.forkNestLevelKey = (nestKey != null) ? nestKey : DEFAULT_FORK_NEST_LEVEL_KEY;
		this.joinVerticesKey = (joinKey != null) ? joinKey : DEFAULT_JOIN_VERTICES_KEY;
		this.reachableVerticesKey = (reachableVertexKey != null) ? reachableVertexKey
				: DEFAULT_REACHABLE_VERTICES_KEY;
		this.forkVerticesKey = (forkKey != null) ? forkKey : DEFAULT_FORK_VERTICES_KEY;

		/* Initialize the Decorators */
		this.nestLevelDecorator = new WatchPointDecorator<String, Integer>(this.forkNestLevelKey,
				UserData.SHARED);
		this.joinVerticesDecorator = new WatchPointDecorator<String, Set<Vertex>>(
				this.joinVerticesKey, UserData.SHARED);
		this.reachableVerticesDecorator = new WatchPointDecorator<String, Set<Vertex>>(
				this.reachableVerticesKey, UserData.SHARED);
		this.forkVerticesDecorator = new WatchPointDecorator<String, Set<Vertex>>(
				this.forkVerticesKey, UserData.SHARED);
	}

	// ========================================================================
	// ===================== ACCESSOR METHODS
	// ========================================================================

	/**
	 * Get the Graph
	 * 
	 * @return the Graph
	 */
	public IWatchPointGraph getGraph()
	{
		return this.graph;
	}

	/**
	 * Get the fork vertices in the graph
	 * 
	 * @return the Fork Vertices
	 */
	public Set<Vertex> getForkVertices()
	{
		return this.forkVertices;
	}

	/**
	 * Get the join vertices in the graph
	 * 
	 * @return The Join Vertices
	 */
	public Set<Vertex> getJoinVertices()
	{
		return this.joinVertices;
	}

	/**
	 * Get the Join Vertices for the Fork Vertex
	 * 
	 * @param forkVertex
	 *            The Fork Vertex
	 * 
	 * @return the set of paired join vertices that the fork vertex reaches.
	 */
	public Set<Vertex> getJoinVertices(Vertex forkVertex)
	{
		return this.joinVerticesDecorator.getValue(forkVertex);
	}

	/**
	 * Get the reachable Vertices for the Fork Vertex
	 * 
	 * @param forkVertex
	 *            The Fork Vertex
	 * 
	 * @return the set of vertices that the Fork Vertex reaches (excluding the Join Vertices).
	 */
	public Set<Vertex> getReachableVertices(Vertex forkVertex)
	{
		return this.reachableVerticesDecorator.getValue(forkVertex);
	}

	/**
	 * Get the Join Vertices for the Fork Edge
	 * 
	 * @param forkEdge
	 *            The outgoing edge from the Fork Vertex
	 * 
	 * @return the set of paired join vertices that the outgoing edge reaches.
	 */
	public Set<Vertex> getJoinVertices(Edge forkEdge)
	{
		return this.joinVerticesDecorator.getValue(forkEdge);
	}

	/**
	 * Get the reachable Vertices for the Fork Edge
	 * 
	 * @param forkEdge
	 *            The outgoing edge from the Fork Vertex
	 * 
	 * @return the set of vertices that the outgoing edge reaches (excluding the Join Vertex).
	 */
	public Set<Vertex> getReachableVertices(Edge forkEdge)
	{
		return this.reachableVerticesDecorator.getValue(forkEdge);
	}

	/**
	 * Get the set of Fork Vertices paired with the Join Vertex
	 * 
	 * @param joinVertex
	 *            The Join Vertex
	 * 
	 * @return The set of Fork vertices that are paired with the Join Vertex
	 */
	public Set<Vertex> getForkVertices(Vertex joinVertex)
	{
		return this.forkVerticesDecorator.getValue(joinVertex);
	}

	/**
	 * Get the Nested Level of the Fork Vertex
	 * 
	 * @param forkVertex
	 *            The Fork Vertex
	 * 
	 * @return The nested level of the fork Vertex (1 indicates no nesting).
	 */
	public int getForkNestLevel(Vertex forkVertex)
	{
		return this.nestLevelDecorator.getValue(forkVertex);
	}

	// ========================================================================
	// ===================== FORK PATH LABELING
	// ========================================================================

	/**
	 * Label all Fork Vertices in the graph with the nest level. Label all Fork Transitions with
	 * their join vertices and the reachable nodes for the transition.
	 * 
	 * @param graph
	 *            The Graph to label.
	 */
	public void labelForkPaths(IWatchPointGraph graph)
	{
		this.initialize(graph);

		for (Vertex forkVertex : this.forkVertices)
		{
			Set<Vertex> reachableVertices = this.reachableVerticesDecorator.getValue(forkVertex);
			Set<Vertex> pairedJoinVertices = this.joinVerticesDecorator.getValue(forkVertex);

			Set<?> edges = forkVertex.getOutEdges();
			for (Object edgeObject : edges)
			{
				Edge forkEdge = (Edge) edgeObject;
				this.labelForkTransition(forkVertex, forkEdge);

				// Collect reachable vertices and join vertices of each transition for the fork
				reachableVertices.addAll(this.reachableVerticesDecorator.getValue(forkEdge));
				pairedJoinVertices.addAll(this.joinVerticesDecorator.getValue(forkEdge));
			}

			// Update the Join Vertices with the newly paired Fork Vertex
			for (Vertex joinVertex : pairedJoinVertices)
			{
				this.forkVerticesDecorator.getValue(joinVertex).add(forkVertex);
			}
		}
	}

	/**
	 * Initialize the state of the labeler, clearing the list of Fork Nodes
	 */
	protected void initialize(IWatchPointGraph graph)
	{
		this.graph = graph;
		this.forkVertices = new HashSet<Vertex>();
		this.joinVertices = new HashSet<Vertex>();

		/* Gather all Fork and Join Vertices, and clear / initialize the decorated values */
		for (Object vertexObject : graph.getVertices())
		{
			Vertex vertex = (Vertex) vertexObject;
			Object element = WatchPointGraphUtils.getElement(vertex);
			if (element instanceof Fork)
			{
				this.forkVertices.add(vertex);
				this.nestLevelDecorator.setValue(1, vertex);
				this.joinVerticesDecorator.setValue(new HashSet<Vertex>(), vertex);
				this.reachableVerticesDecorator.setValue(new HashSet<Vertex>(), vertex);

				Set<?> edges = vertex.getOutEdges();
				for (Object edgeElement : edges)
				{
					Edge edge = (Edge) edgeElement;
					this.joinVerticesDecorator.setValue(new HashSet<Vertex>(), edge);
					this.reachableVerticesDecorator.setValue(new HashSet<Vertex>(), edge);
				}
			}
			else if (element instanceof Join)
			{
				this.joinVertices.add(vertex);
				this.forkVerticesDecorator.setValue(new HashSet<Vertex>(), vertex);
			}
		}
	}

	/**
	 * Label the input Fork Transition with the join vertices it reaches, and the list of vertices
	 * it reaches before the join vertices. Updates the nest level labels for the Fork nodes in the
	 * process.
	 * 
	 * @param forkVertex
	 *            The Fork Vertex
	 * @param forkEdge
	 *            The transition from the Fork Vertex
	 */
	protected void labelForkTransition(Vertex forkVertex, Edge forkEdge)
	{
		Vertex firstVertex = forkEdge.getOpposite(forkVertex);
		VertexVisitorState state = new VertexVisitorState(forkVertex, forkEdge);
		visitVertex(firstVertex, state);

		this.joinVerticesDecorator.setValue(state.joinVertices, forkEdge);
		this.reachableVerticesDecorator.setValue(state.reachableVertices, forkEdge);
	}

	/*
	 * Recursive DFS to find the reachable vertices and all possible join vertices.
	 */
	private void visitVertex(Vertex vertex, VertexVisitorState state)
	{
		// Check if the Vertex has already been visited (loop)
		if (state.reachableVertices.contains(vertex))
			return;

		// Check if the Vertex is the starting Fork node (invalid loop)
		if (vertex.equals(state.forkVertex))
			return;

		Object element = WatchPointGraphUtils.getElement(vertex);
		if (element instanceof Join)
		{
			if (state.currentNestLevel == state.startNestLevel)
			{ // Found the end of the path
				state.joinVertices.add(vertex);
				return;
			}
			else
			{ // Found a nested join
				state.currentNestLevel--;
			}
		}
		else if (element instanceof Fork)
		{ // Found a nested fork
			state.currentNestLevel++;
			int forkNestLevel = this.nestLevelDecorator.getValue(vertex);
			if (state.currentNestLevel > forkNestLevel)
				this.nestLevelDecorator.setValue(state.currentNestLevel, vertex);
		}

		// Mark the vertex as visited
		state.reachableVertices.add(vertex);

		int currentNestLevel = state.currentNestLevel;
		for (Object vertexObject : vertex.getSuccessors())
		{
			Vertex successor = (Vertex) vertexObject;
			this.visitVertex(successor, state);
			state.currentNestLevel = currentNestLevel;
		}
	}

	/*
	 * Internal class that maintains state used during the recursive DFS
	 */
	private class VertexVisitorState
	{
		private Set<Vertex> joinVertices;

		private Set<Vertex> reachableVertices;

		private Vertex forkVertex;

		private int currentNestLevel;

		private int startNestLevel;

		/*
		 * Constructor
		 */
		VertexVisitorState(Vertex forkVertex, Edge forkEdge)
		{
			this.forkVertex = forkVertex;
			this.joinVertices = ForkPathLabeler.this.joinVerticesDecorator.getValue(forkEdge);
			this.reachableVertices = ForkPathLabeler.this.reachableVerticesDecorator.getValue(forkEdge);
			this.startNestLevel = ForkPathLabeler.this.nestLevelDecorator.getValue(forkVertex);
			this.currentNestLevel = startNestLevel;
		}
	}
}
