/*
 * Created on Jul 16, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.graph;

import java.util.HashSet;
import java.util.Set;

import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;

import edu.uci.ics.jung.graph.DirectedEdge;
import edu.uci.ics.jung.graph.Vertex;
import edu.uci.ics.jung.graph.impl.DirectedSparseEdge;
import edu.uci.ics.jung.graph.impl.SimpleDirectedSparseVertex;
import edu.uci.ics.jung.utils.GraphUtils;
import edu.uci.ics.jung.utils.UserData;

/**
 * Factory class that creates Graph objects.
 * 
 * @author Vijay Silva
 */
public class GraphFactory
{
	/**
	 * Constructor
	 */
	protected GraphFactory()
	{
	}

	/**
	 * Create a new Graph Factory instance
	 * 
	 * @return The GraphFactory instance
	 */
	public static GraphFactory createFactory()
	{
		return new GraphFactory();
	}

	/**
	 * Creates a new Graph for the input object. Every Graph element (transition, node, graph)
	 * contains the key: ELEMENT_KEY that contains the object that the element represents.
	 * 
	 * @param objectToGraph
	 *            The object for which a graph must be constructed
	 * 
	 * @return The Graph reflecting the object
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.graph.IWatchPointGraph#ELEMENT_KEY
	 */
	public IWatchPointGraph createGraph(Object objectToGraph)
	{
		IWatchPointGraph graph = null;

		// Create the graph instance
		if (objectToGraph instanceof WorkflowTemplate)
		{
			graph = new WorkflowTemplateGraph();
			this.buildGraph((WorkflowTemplate) objectToGraph, graph);
		}
		else if (objectToGraph instanceof NodeDefinition)
		{
			graph = new NodeDefinitionGraph();
			this.buildGraph((NodeDefinition) objectToGraph, graph);
		}
		else
		{
			String objDesc = (objectToGraph == null) ? "a null object." : "an object of type: "
					+ objectToGraph.getClass().getName() + ".";
			throw new IllegalArgumentException("Cannot build a graph for " + objDesc);
		}

		return graph;
	}

	/**
	 * Create a 'Reverse' Graph in which the direction of every edge is reversed.
	 * 
	 * @param <T>
	 *            The Type of the graph being reversed.
	 * @param graph
	 *            The graph which needs to be reversed.
	 * 
	 * @return The reversed graph
	 */
	@SuppressWarnings("unchecked")
	public <T extends IWatchPointGraph> T createReverseGraph(T graph)
	{
		T graphCopy = (T) graph.copy();

		Set<DirectedEdge> edges = graphCopy.getEdges();
		Set<DirectedEdge> reversedEdges = new HashSet<DirectedEdge>();

		for (DirectedEdge edge : edges)
		{
			reversedEdges.add(this.createEdge(edge.getDest(), edge.getSource(), null));
		}

		GraphUtils.removeEdges(graphCopy, edges);
		GraphUtils.addEdges(graphCopy, reversedEdges);

		return graphCopy;
	}

	/*
	 * Build a Graph given a NodeElementContainer. Used for both Templates and Node Definitions.
	 */
	private void buildGraph(NodeElementContainer nodeContainer, IWatchPointGraph graph)
	{
		NodeElement[] nodes = nodeContainer.getNodeElements();
		int nodeCount = (nodes != null) ? nodes.length : 0;

		// Store the Node Element Container as the Graph data
		graph.addUserDatum(IWatchPointGraph.ELEMENT_KEY, nodeContainer, UserData.SHARED);

		// Create all the vertices
		Vertex[] vertices = new Vertex[nodeCount];
		for (int i = 0; i < nodeCount; i++)
		{
			vertices[i] = this.createVertex(nodes[i]);
			graph.addVertex(vertices[i]);
		}

		// Create the edges
		for (int i = 0; i < nodeCount; i++)
		{
			Transition[] transitions = nodes[i].getTransitions();
			int transitionCount = (transitions != null) ? transitions.length : 0;

			for (int t = 0; t < transitionCount; t++)
			{
				Transition transition = transitions[t];
				String nodeName = transition.getTo();
				Vertex destination = this.findVertex(graph, nodeName);
				if (!destination.isSuccessorOf(vertices[i]))
					graph.addEdge(this.createEdge(vertices[i], destination, transition));
			}
		}
	}

	/*
	 * Creates a Vertex that can be used in a WatchPoint Graph
	 */
	private Vertex createVertex(Object element)
	{
		Vertex vertex = new SimpleDirectedSparseVertex();
		vertex.addUserDatum(IWatchPointGraph.ELEMENT_KEY, element, UserData.SHARED);

		return vertex;
	}

	/*
	 * Creates a Directed Edge that can be used in a WatchPoint Graph
	 */
	private DirectedEdge createEdge(Vertex source, Vertex destination, Object edgeElement)
	{
		DirectedEdge edge = new DirectedSparseEdge(source, destination);
		if (edgeElement != null)
			edge.addUserDatum(IWatchPointGraph.ELEMENT_KEY, edgeElement, UserData.SHARED);

		return edge;
	}

	/*
	 * Find a Vertex using the 'name' key
	 */
	@SuppressWarnings("unchecked")
	private Vertex findVertex(IWatchPointGraph graph, String name)
	{
		Vertex result = null;

		for (Vertex vertex : ((Set<Vertex>) graph.getVertices()))
		{
			NodeElement node = (NodeElement) vertex.getUserDatum(IWatchPointGraph.ELEMENT_KEY);
			String vertexElementName = (node != null) ? node.getName() : null;
			if (vertexElementName != null && vertexElementName.equals(name))
			{
				result = vertex;
				break;
			}
		}

		return result;
	}
}
