/*
 * Created on Jul 10, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.impl;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.log4j.Logger;

import com.tandbergtv.watchpoint.studio.validation.IValidationRule;

/**
 * Utility class that provides methods to read the configured validation rules for each type of
 * Object that is specified in a properties file.
 * 
 * @author Vijay Silva
 */
public final class ValidationRuleReader
{
	private static final String RESOURCE_NAME = "/validation-rules.properties";

	private static final Logger logger = Logger.getLogger(ValidationRuleReader.class);

	/*
	 * Constructor
	 */
	private ValidationRuleReader()
	{
	}

	/**
	 * Read the list of validation rules for each Object type from a configuration file. Returns a
	 * map of the object class to the list of Validation Rules for that object type.
	 * 
	 * @return Map of the object type to the validation rules for the object type.
	 * 
	 * @throws ValidationRuleConfiguratonException
	 *             The validation rule configuration is invalid or cannot be accessed/read. If the
	 *             configuration is partially correct, the partial configuration is returned with
	 *             the exception.
	 */
	public static Map<Class<?>, List<IValidationRule<?>>> readValidationRules()
			throws ValidationRuleConfiguratonException
	{

		Map<Class<?>, List<IValidationRule<?>>> ruleMap = new HashMap<Class<?>, List<IValidationRule<?>>>();
		boolean invalidConfiguration = false;

		InputStream inStream = null;
		try
		{
			Properties properties = new Properties();
			inStream = ValidationRuleReader.class.getResourceAsStream(RESOURCE_NAME);
			properties.load(inStream);

			for (Object key : properties.keySet())
			{
				String entityClassName = (String) key;
				String value = properties.getProperty(entityClassName);

				try
				{
					String[] ruleClassNames = value.split(",");
					int ruleClassCount = (ruleClassNames != null) ? ruleClassNames.length : 0;

					Class<?> entityClass = Class.forName(entityClassName);
					List<IValidationRule<?>> rules = new ArrayList<IValidationRule<?>>();
					for (int i = 0; i < ruleClassCount; i++)
					{
						String name = (ruleClassNames[i] != null) ? ruleClassNames[i].trim() : "";
						if (name.length() > 0)
						{
							Class<?> ruleClass = Class.forName(name);
							IValidationRule<?> rule = (IValidationRule<?>) ruleClass.newInstance();
							rules.add(rule);
						}
					}

					ruleMap.put(entityClass, rules);
				}
				catch (Exception ex)
				{
					invalidConfiguration = true;
					String msg = "Failed to read and load the validation rules for entity: "
							+ entityClassName;
					logger.error(msg, ex);
				}
			}
		}
		catch (Exception ex)
		{
			String msg = "Failed to load the configuration of the Validation Rules.";
			throw new ValidationRuleConfiguratonException(msg, ruleMap, ex);
		}
		finally
		{
			try
			{
				if (inStream != null)
					inStream.close();
			}
			catch (IOException ioe)
			{
				logger.warn("Failure when closing the stream.", ioe);
			}

		}

		if (invalidConfiguration)
		{
			String msg = "Failed to load some of the configuration of the Validation Rules.";
			throw new ValidationRuleConfiguratonException(msg, ruleMap);
		}

		return ruleMap;
	}
}
