/*
 * Created on Jun 21, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess.jpa;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.persistence.NoResultException;
import javax.persistence.Query;

import com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.TemplateManualTaskNodeDTO;
import com.tandbergtv.watchpoint.studio.dto.TemplateMessageDTO;
import com.tandbergtv.watchpoint.studio.dto.TemplateMessageDTO.TemplateMessageType;
import com.tandbergtv.watchpoint.studio.dto.TemplateSubProcessDTO;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;

/**
 * The JPA Data Access Implementation for the WorkflowTemplateDTO entity.
 * 
 * @author Vijay Silva
 */
public class WorkflowTemplateDTOPDAO extends PersistenceDAO<WorkflowTemplateDTO, Long> implements
		WorkflowTemplateDTODAI
{
	/**
	 * Constructor
	 * 
	 * @param persistenceContext
	 *            The Persistence Context to use for the Data Access operations.
	 */
	public WorkflowTemplateDTOPDAO(JPAPersistenceContext persistenceContext)
	{
		super(WorkflowTemplateDTO.class, persistenceContext);
	}

	/**
	 * Optimizing the query to find all Templates.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.jpa.PersistenceDAO#findAll()
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> findAll()
	{
		String queryName = "WorkflowTemplateDTO.All";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		return query.getResultList();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#findByCurrentVersion()
	 */
	public List<WorkflowTemplateDTO> findByCurrentVersion()
	{
		String queryName = "WorkflowTemplateDTO.ByCurrentVersion";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		return this.buildTemplates(query.getResultList());
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#findByTemplateUsage(String)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> findByTemplateUsage(String subProcessName)
	{
		String queryName = "WorkflowTemplateDTO.ByTemplateUsage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("subProcessName", subProcessName);
		return query.getResultList();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#findByMessageUsage(Collection<String>, TemplateMessageType)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> findByMessageUsage(Collection<String> messageUIDs, NodeDefinitionType type)
	{
		String queryName = "WorkflowTemplateDTO.ByMessageUsage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("messageUID", messageUIDs);
		query.setParameter("nodeType", type);
		return query.getResultList();
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#findBySuperStateUsage(String,String)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> findBySuperStateUsage(String nodeDefinitionName, String systemID)
	{
		String queryName = "WorkflowTemplateDTO.BySuperStateUsage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("nodeDefinitionName", nodeDefinitionName);
		query.setParameter("systemID", systemID);
		query.setParameter("nodeType", TemplateMessageType.SuperState);
		return query.getResultList();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#findByName(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	public List<WorkflowTemplateDTO> findByName(String name)
	{
		String queryName = "WorkflowTemplateDTO.ByName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);

		List<WorkflowTemplateDTO> result = null;
		try {
			result = (List<WorkflowTemplateDTO>) query.getResultList();
		} catch (NoResultException e) {
			// No template found - return null
		}
		
		return result;
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#findByProjectName(java.lang.String)
	 */
	@SuppressWarnings("unchecked")
	public List<WorkflowTemplateDTO> findByProjectName(String projectName)
	{
		String queryName = "WorkflowTemplateDTO.ByProjectName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("projectName", projectName);

		List<WorkflowTemplateDTO> result = null;
		try {
			result = (List<WorkflowTemplateDTO>) query.getResultList();
		} catch (NoResultException e) {
			// No template found - return null
		}
		
		return result;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#getTemplateIdByName(java.lang.String)
	 */
	public Long getTemplateIdByName(String name)
	{
		String queryName = "WorkflowTemplateDTO.IdByName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);

		return (Long) query.getSingleResult();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#getTemplateIdByNameForOldestVersion(java.lang.String)
	 */
	public Long getTemplateIdByNameForOldestVersion(String name)
	{
		String queryName = "WorkflowTemplateDTO.IdByNameForOldestVersion";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);

		return (Long) query.getSingleResult();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#getCurrentVersionNumberByName(String)
	 */
	public Integer getCurrentVersionNumberByName(String name)
	{
		String queryName = "WorkflowTemplateDTO.CurrentVersionByName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);
		return (Integer) query.getSingleResult();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#getCountByName(java.lang.String)
	 */
	public int getCountByName(String name)
	{
		String queryName = "WorkflowTemplateDTO.CountByName";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("name", name);
		Long count = (Long) query.getSingleResult();
		return count.intValue();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#getCountByResourceGroupUsage(String)
	 */
	public int getCountByResourceGroupUsage(String resourceGroupName)
	{
		String queryName = "WorkflowTemplateDTO.CountByResourceGroupUsage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("resourceGroupName", resourceGroupName);
		Long count = (Long) query.getSingleResult();
		return count.intValue();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#deleteNodeDefinitionReferences(long)
	 */
	public void deleteNodeDefinitionReferences(long nodeDefinitionId)
	{
		String sqlQuery = "DELETE from TB_NODEDEFINITIONTEMPLATEMAP WHERE NODEDEFINITIONID = :nodeDefinitionId";
		Query query = getEntityManager().createNativeQuery(sqlQuery);
		query.setParameter("nodeDefinitionId", nodeDefinitionId);
		query.executeUpdate();
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/*
	 * Create a WorkflowTemplateDTO entity from the array of column values.
	 */
	private List<WorkflowTemplateDTO> buildTemplates(List<?> data)
	{
		List<WorkflowTemplateDTO> templates = new ArrayList<WorkflowTemplateDTO>();

		if (data != null)
		{
			for (Object dataObject : data)
			{
				Object[] columns = (Object[]) dataObject;
				WorkflowTemplateDTO template = new WorkflowTemplateDTO();
				for (int i = 0; i < columns.length; i++)
				{
					switch (i)
					{
						case 0:
							template.setId((Long) columns[0]);
							break;
						case 1:
							template.setName((String) columns[1]);
							break;
					}
				}

				templates.add(template);
			}
		}

		return templates;
	}

	@Override
	public WorkflowTemplateDTO findByPath(String path) {
		String queryName = "WorkflowTemplateDTO.ByPath";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("path", path);

		WorkflowTemplateDTO result = null;
		try {
			result = (WorkflowTemplateDTO) query.getSingleResult();
		} catch (javax.persistence.NoResultException e) {
			// No template found - return null
		}
		
		return result;
	}
	
	/**
	 * Need to remove all Parameters and Messages that are not used anymore.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.jpa.PersistenceDAO#update(com.tandbergtv.watchpoint.studio.dto.IPersistable)
	 */
	@Override
	public WorkflowTemplateDTO update(WorkflowTemplateDTO entity) {
		
		// Delete all the previous node definitions to avoid orphans.
		WorkflowTemplateDTO previous = this.find(entity.getKey());
		for (TemplateMessageDTO nodeDefinition : previous.getMessages()) {
			this.getEntityManager().remove(nodeDefinition);
		}
		for (TemplateSubProcessDTO subProcess : previous.getSubprocesses()) {
			this.getEntityManager().remove(subProcess);
		}
		for (TemplateManualTaskNodeDTO manTaskNode : previous.getManualTaskNodes()) {
			this.getEntityManager().remove(manTaskNode);
		}
		
		return super.update(entity);
	}


	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#findByResourceTypeUsage(String)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> findByResourceTypeUsage(String systemID) {
		String queryName = "WorkflowTemplateDTO.ByResourceTypeUsage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("systemID", systemID);
		return query.getResultList();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI#findByResourceTypeUsage(String)
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<WorkflowTemplateDTO> findByResourceGroupUsage(String groupName) {
		String queryName = "WorkflowTemplateDTO.ByResourceGroupUsage";
		Query query = this.getEntityManager().createNamedQuery(queryName);
		query.setParameter("resourceGroupName", groupName);
		return query.getResultList();
	}
}
