/*
 * Created on Sep 28, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart;

import static com.tandbergtv.watchpoint.studio.dto.IPersistable.DEFAULT_ID;
import static com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType.MessageNode;
import static com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType.SuperState;
import static com.tandbergtv.watchpoint.studio.service.ServiceFactory.createFactory;
import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.NODE_DEFN_SEID;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.ID_RESOURCE_TYPE;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.NODEDEF_EDITOR_ID;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.getAdapterFromStream;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.handleException;
import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.ADAPTOR_FOLDER_PATH;
import static javax.xml.xpath.XPathConstants.STRING;
import static org.eclipse.jface.dialogs.MessageDialog.openConfirm;
import static org.eclipse.jface.dialogs.MessageDialog.openError;
import static org.eclipse.swt.widgets.Display.getCurrent;

import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathFactory;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerComparator;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.navigator.CommonViewer;
import org.eclipse.ui.navigator.resources.ProjectExplorer;
import org.eclipse.ui.statushandlers.StatusManager;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import com.tandbergtv.watchpoint.studio.dto.AdaptorType;
import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ReferenceDeletionException;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.ResourceTypeEditorInput;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.util.ServiceErrors;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer;
import com.tandbergtv.watchpoint.studio.ui.wizard.MessageWizard;
import com.tandbergtv.watchpoint.studio.ui.xml.NodeDefinitionDomAdapter;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;

/**
 * The Section Part displaying the list of all Messages
 * 
 * @author Vijay Silva
 */
public class MessageMasterSectionPart extends ResourceTypeSectionPart
{
	private static final int MESSAGE_UID_INDEX = 0;

	private static final int MESSAGE_NAME_INDEX = 1;
	
	private static final int MESSAGE_TIMEOUT_INDEX = 2;

	TableViewer messagesTableViewer;

	Table messagesTable;

	TableColumn nameColumn, uidColumn, timeoutColumn;

	Composite buttonComposite;

	Button createButton, deleteButton;

	Label createInfoLabel, infoLabel;
	
	private static final String INITIAL_SINGLE_NODE_GPD_XML = "initialSingleNodeGPD.xml";
	
	private static final String INITIAL_NODE_DEFINITION_XML = "initialNodeDefinition.xml";
	
	private static final String AUTO_TASK_NODE_NAME = "//node[1]/@name";
	
	private Map<String, String> nodesTimeout;

	// ========================================================================
	// ===================== CONSTRUCTORS
	// ========================================================================

	/**
	 * @param parentPart
	 *            The Part that contains this composite
	 * @param parent
	 *            The Parent Composite containing the section to be created
	 * @param toolkit
	 *            The toolkit to create the section with
	 * @param style
	 *            The style bits to create the section with
	 */
	public MessageMasterSectionPart(IEditorPart parentPart, Composite parent, FormToolkit toolkit,
			int style) {
		super(parentPart, parent, toolkit, style);
	}

	// ========================================================================
	// ===================== USER CONTROL INITIALIZATION
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#createSectionClientControls(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void createSectionClientControls(IManagedForm managedForm) {
		FormToolkit toolkit = managedForm.getToolkit();

		/* The Messages Table and Buttons */
		this.initializeMessagesTable(managedForm);
		this.buttonComposite = toolkit.createComposite(this.sectionClient);
		toolkit.paintBordersFor(this.buttonComposite);
		this.createButton = toolkit.createButton(this.buttonComposite, "Create...", SWT.PUSH);
		this.deleteButton = toolkit.createButton(this.buttonComposite, "Delete", SWT.PUSH);

		/* Add the listeners for the buttons */
		this.createButton.addSelectionListener(new ButtonPressListener());
		this.deleteButton.addSelectionListener(new ButtonPressListener());

		/* The Labels */
		String createInfoText = "* Messages cannot be created if Adaptor Type is not set.";
		this.createInfoLabel = toolkit.createLabel(this.sectionClient, createInfoText);

		String infoText = "* Messages that are created or deleted are instantly saved, "
				+ System.getProperty("line.separator")
				+ "the Resource Type does not need to be saved explicitly.";
		this.infoLabel = toolkit.createLabel(this.sectionClient, infoText);
	}

	/*
	 * Initialize the Messages Table
	 */
	private void initializeMessagesTable(IManagedForm managedForm) {
		FormToolkit toolkit = managedForm.getToolkit();

		int style = SWT.SINGLE | SWT.FULL_SELECTION | SWT.H_SCROLL | SWT.V_SCROLL;
		this.messagesTable = toolkit.createTable(this.sectionClient, style);
		toolkit.paintBordersFor(this.messagesTable);

		this.uidColumn = new TableColumn(messagesTable, SWT.LEFT, 0);
		this.uidColumn.setText("Message UID");
		this.uidColumn.setToolTipText("Message UID");
		this.uidColumn.setResizable(true);
		this.uidColumn.setMoveable(false);
		this.uidColumn.setWidth(110);

		this.nameColumn = new TableColumn(messagesTable, SWT.LEFT, 1);
		this.nameColumn.setText("Message Name");
		this.nameColumn.setToolTipText("Message Name");
		this.nameColumn.setResizable(true);
		this.nameColumn.setMoveable(false);
		this.nameColumn.setWidth(220);

		this.timeoutColumn = new TableColumn(messagesTable, SWT.LEFT, 2);
		this.timeoutColumn.setText("Time Out");
		this.timeoutColumn.setToolTipText("Time Out");
		this.timeoutColumn.setResizable(true);
		this.timeoutColumn.setMoveable(false);
		this.timeoutColumn.setWidth(198);

		this.messagesTable.setHeaderVisible(true);
		this.messagesTable.setSortColumn(this.nameColumn);
		this.messagesTable.setSortDirection(SWT.DOWN);
		this.messagesTable.setTopIndex(0);
		this.messagesTable.setLinesVisible(true);

		this.messagesTableViewer = new TableViewer(messagesTable);
		this.messagesTableViewer.setLabelProvider(new MessageLabelProvider());
		this.messagesTableViewer.setContentProvider(new MessageContentProvider());
		this.messagesTableViewer.addSelectionChangedListener(new MessageSelectionChangedListener());
		this.messagesTableViewer.setComparator(new ViewerComparator());
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#initializeSectionClientLayout(org.eclipse.ui.forms.IManagedForm)
	 */
	@Override
	protected void initializeSectionClientLayout(IManagedForm managedForm) {
		/* Set the Layout for the Section Client in the Form */
		this.sectionClient.setLayout(this.createGridLayout(1, false));

		/* Set the Layout data for the widgets */
		GridData gridData = new GridData(SWT.FILL, SWT.FILL, true, false);

		gridData.minimumHeight = 150;
		gridData.heightHint = 300;
		gridData.widthHint = 500;
		this.messagesTable.setLayoutData(gridData);

		this.buttonComposite.setLayoutData(this.createGridData());

		RowLayout rowLayout = new RowLayout(SWT.HORIZONTAL);
		rowLayout.fill = true;
		rowLayout.pack = false;
		rowLayout.marginTop = rowLayout.marginBottom = 0;
		rowLayout.marginLeft = rowLayout.marginRight = 0;
		rowLayout.spacing = 5;
		this.buttonComposite.setLayout(rowLayout);

		this.createButton.setLayoutData(new RowData());
		this.deleteButton.setLayoutData(new RowData());

		int labelWidth = this.messagesTable.getSize().x;
		gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
		gridData.widthHint = labelWidth;
		gridData.horizontalSpan = 2;
		this.createInfoLabel.setLayoutData(gridData);

		gridData = new GridData(SWT.FILL, SWT.BEGINNING, true, false);
		gridData.widthHint = labelWidth;
		gridData.horizontalSpan = 2;
		this.infoLabel.setLayoutData(gridData);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#populateSectionData()
	 */
	@Override
	protected void populateSectionData() {
		populateNodesTimeoutMap();
		this.messagesTableViewer.setInput(this.input);

		ISelection selection = this.messagesTableViewer.getSelection();
		if (!selection.isEmpty())
			this.getManagedForm().fireSelectionChanged(this, selection);
	}
	
	private void populateNodesTimeoutMap() {
		this.nodesTimeout = new HashMap<String, String>();
		ResourceTypeEditorInput input = (ResourceTypeEditorInput) this.input;
		ResourceType resourceType = input.getResourceType();
		
		for (NodeDefinitionDTO nodeDTO : resourceType.getNodes()) {
			if (nodeDTO.getType().equals(NodeDefinitionType.MessageNode)) {
				getNodeDefinitionUID(nodeDTO.getXml());
			}
		}
	}
	
	private void getNodeDefinitionUID(String xmlString) {
		try {
			Document doc = XMLDocumentUtility.loadXml(xmlString);
			XPath xpath = XPathFactory.newInstance().newXPath();
			Element e = doc.getDocumentElement();
			
			String sendNodeUID = (String) xpath.evaluate("//action/send", e, STRING);
			sendNodeUID = sendNodeUID.trim();

			String receiveNodeUID = (String) xpath.evaluate("//action/receive", e, STRING);
			receiveNodeUID = receiveNodeUID.trim();

			String timeOut = (String) xpath.evaluate("//event/create-timer/@duedate", e, STRING);

			if (timeOut != null && !timeOut.isEmpty()) {
				String key = sendNodeUID.isEmpty() ? receiveNodeUID : sendNodeUID;
				nodesTimeout.put(key, timeOut);
			}
		} catch (Exception e) {
			Status status = new Status(IStatus.ERROR, "WatchPoint Studio", IStatus.ERROR, "Error fetching UID", e);
			StatusManager.getManager().handle(status, StatusManager.SHOW);
		}
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpart.ResourceTypeSectionPart#setWidgetPermissions()
	 */
	@Override
	protected void setWidgetPermissions() {
		boolean isEditable = this.isInputEditable();
		this.createButton.setEnabled(isEditable);
		this.deleteButton.setEnabled(isEditable);
		this.updateButtonStatus();
	}

	// ========================================================================
	// ===================== ACTIONS FOR THE TABLE
	// ========================================================================

	/*
	 * Update the Status of the Create and Delete Buttons
	 */
	private void updateButtonStatus() {
		if (!this.isInputEditable())
			return;

		this.createButton.setEnabled(this.isCreateAllowed());
		this.deleteButton.setEnabled(this.isDeleteAllowed());
	}

	/*
	 * Check if the Create Action is allowed
	 */
	private boolean isCreateAllowed() {
		if (!this.isInputEditable())
			return false;

		/* Create is allowed if the adaptor type is not none */
		ResourceTypeEditorInput adapter = null;
		AdaptorType adaptorType = null;

		adapter = (ResourceTypeEditorInput) this.input.getAdapter(ResourceTypeEditorInput.class);
		if (adapter != null)
			adaptorType = adapter.getResourceType().getAdaptorType();

		return (adaptorType != null && adaptorType != AdaptorType.NONE);
	}

	/*
	 * Check if the Delete Action is allowed
	 */
	private boolean isDeleteAllowed() {
		if (!this.isInputEditable())
			return false;

		IStructuredSelection selection = this.getTableSelection();
		Object selectedObject = selection.getFirstElement();

		return (selectedObject != null);
	}

	/*
	 * Get the Selection for the table
	 */
	private IStructuredSelection getTableSelection() {
		ISelection selection = this.messagesTableViewer.getSelection();
		return (IStructuredSelection) selection;
	}

	// ========================================================================
	// ===================== ACTIONS FOR THE BUTTONS
	// ========================================================================

	/*
	 * Create a New Message for the Resource Type
	 */
	private void createNewMessage() {
		if (!this.isCreateAllowed()) {
			this.updateButtonStatus();
			return;
		}

		// Show the Message Wizard and create a new Message
		ResourceTypeEditorInput adapter = (ResourceTypeEditorInput) this.input.getAdapter(ResourceTypeEditorInput.class);
		
		if (adapter == null || adapter.getPersistedResourceType() == null)
			return;

		MessageWizard messageWizard = new MessageWizard(adapter.getPersistedResourceType());
		WizardDialog dialog = new WizardDialog(getSection().getShell(), messageWizard);
		
		dialog.open();

		if (dialog.getReturnCode() != Window.OK)
			return;

		String timeout = messageWizard.getTimeout();
		Message message = messageWizard.getMessage();

		adapter.addCreatedMessage(message);

		ResourceType resourceType = adapter.getPersistedResourceType();

		try {
			NodeDefinitionDTO dto = MessageMasterSectionPart.createInitialDTO(resourceType, message, timeout);
			createFactory().createNodeDefinitionService().createNodeDefinitionInFileSystem(dto);

			IResourceTypeService rtService = createFactory().createResourceTypeService();

			rtService.updateResourceTypeInFileSystem(resourceType);

			ResourceType updated = rtService.getResourceTypeFromFileSystem(resourceType.getProjectName());

			if (updated != null) {
				adapter.updatePersistedResourceType(updated);
			} else {
				Utility.reportError("Error updating resource type", null);
			}

			refreshViews();
		} catch (ServiceException e) {
			handleException(ServiceErrors.getInstance().getProperty(e.getServiceErrorCode().getCode()), e);
		} catch (Exception e) {
			openError(getCurrent().getActiveShell(), "Error", "Failed to create Message Node");
			return;
		}

		this.markStale();
	}

	/*
	 * Creates the initial NodeDefinitionDTO object.
	 * 
	 * @return initial NodeDefinitionDTO object
	 */
	public static NodeDefinitionDTO createInitialDTO(ResourceType rt, Message message, String timeout) throws Exception {
		String xml = createInitialNodeDefinition(rt.getId(), message.getName(), message.getUid(), timeout);
		String gpd = createInitialGpdInfo(message.getName());
		NodeDefinitionDTO dto = Utility.createNodeDefinitionDTO(DEFAULT_ID, message.getName(), xml, gpd);
		
		dto.setType(MessageNode);
		dto.addMessage(message);
		dto.setResourceType(rt);
		dto.setXml(xml);
		
		return dto;
	}
	
	public static String createInitialNodeDefinition(long id, String name, String uid, String timeout) {
		String xml = INITIAL_NODE_DEFINITION_XML;
		SemanticElementFactory factory = new SemanticElementFactory(NODEDEF_EDITOR_ID);
		NodeDefinition node = (NodeDefinition) factory.createById(NODE_DEFN_SEID);
		NodeDefinitionDomAdapter adapter = 
			(NodeDefinitionDomAdapter) getAdapterFromStream(xml, MessageMasterSectionPart.class.getResourceAsStream(xml), factory);
		
		adapter.initialize(node);
		node.setName(name);
		node.setResourceType(id);
		node.setDueDate(timeout);
		node.setUid(uid);
		
		return Utility.getXMLFromDocument(adapter.getNode().getOwnerDocument());
	}
	
	public static String createInitialGpdInfo(String name) throws Exception {
		String gpdFileName = INITIAL_SINGLE_NODE_GPD_XML;

		DocumentBuilderFactory domFactory = DocumentBuilderFactory.newInstance();
		domFactory.setNamespaceAware(true);
		DocumentBuilder builder = domFactory.newDocumentBuilder();
		Document doc = builder.parse(MessageMasterSectionPart.class.getResourceAsStream(gpdFileName));

		XPath xpath = XPathFactory.newInstance().newXPath();
		XPathExpression expr = xpath.compile(AUTO_TASK_NODE_NAME);

		Node nameAttribute = (Node) expr.evaluate(doc, XPathConstants.NODE);
		if (nameAttribute != null) {
			nameAttribute.setNodeValue(name);
		}

		return Utility.getXMLFromDocument(doc);
	}
	
	/*
	 * Delete the selected message from the Resource Type
	 */
	private void deleteSelectedMessage() {
		if (!this.isDeleteAllowed()) {
			this.updateButtonStatus();
			return;
		}

		ResourceTypeEditorInput adapter = null;
		adapter = (ResourceTypeEditorInput) this.input.getAdapter(ResourceTypeEditorInput.class);
		if (adapter == null || adapter.getPersistedResourceType() == null)
			return;

		IStructuredSelection selection = this.getTableSelection();
		INodeDefinitionService nodeDefinitionService = null;
	
		ServiceFactory factory = ServiceFactory.createFactory();
		IResourceTypeService resourceTypeService = factory.createResourceTypeService();
		
		Object selectedObject = selection.getFirstElement();
		
		if (!(selectedObject instanceof MessageInput))
			return;
		
		MessageInput messageInput = (MessageInput) selectedObject;
		Message message = messageInput.getMessage();
		
		String msg = "Are you sure you want to delete message '" + message.getName() + "'?";
		boolean response = MessageDialog.openConfirm(Display.getCurrent().getActiveShell(), "Confirm Delete", msg);
		
		if (!response)
			return;
		
		boolean failedRemovingNode = false;
		ResourceType resourceType = adapter.getResourceType();
		Set<NodeDefinitionDTO> nodes = resourceType.getNodes();
		
		for (NodeDefinitionDTO node: nodes) {
			try {
				NodeDefinition nodeDef = SemanticElementUtil.createNodeDefinition(node.getXml());
				
				if (nodeDef.getNodeType() == SuperState)
					continue;
				
				if (message.getName().equals(nodeDef.getName())){
					nodeDefinitionService = ServiceFactory.createFactory().createNodeDefinitionService();
					nodeDefinitionService.deleteNodeDefinitionInFileSystem(node, false);
					break;
				}
			} catch (ReferenceDeletionException e) {
				String emsg = buildReferenceDeletionErrorMsg(e.getReferences());

				boolean removeReferences = openConfirm(super.getSection().getShell(), "Remove References?", emsg);
				
				if (!removeReferences)
					return;
				
				nodeDefinitionService.deleteNodeDefinition(node.getId(), true);
				resourceType.removeNode(node);
			} catch (ServiceException e) {							
				failedRemovingNode = true;
				handleException(ServiceErrors.getInstance().getProperty(e.getServiceErrorCode().getCode()), e);
				return;//to skid delete message when locked?
            } catch (Exception e) {
				Utility.handleException("Failed creating node definition from xml.", e);
			}
		}

		if (!failedRemovingNode) {
			resourceType.removeMessage(message);

			// Delete adaptor definition for the message
			IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(resourceType.determineNameForProject());
			
			if (project.exists()) {
				try {
					project.getFolder(ADAPTOR_FOLDER_PATH + message.getUid()).delete(true, true, null);
				} catch (CoreException e) {
					handleException("Message deleted but unable to delete message folder from the resource type project.", e);
				}
			}

			resourceTypeService.updateResourceTypeInFileSystem(resourceType);

			ResourceType updated = resourceTypeService.getResourceTypeFromFileSystem(resourceType.getProjectName());
			
			if (updated != null) {
				adapter.updatePersistedResourceType(updated);
			} else {
				Utility.reportError("Error updating resource type", null);
			}
		}

		this.markStale();
		refreshViews();
	}
	
	private String buildReferenceDeletionErrorMsg(List<IWatchPointDTO> references) {
		String message = "";
		String LINE_SEPRATOR = "line.separator";
		Set<String> templateSet = new HashSet<String>();
		Set<String> nodeDefSet = new HashSet<String>();
		/* Add DTO objects to sets to remove duplicate names */
		for (IWatchPointDTO dto : references) {
			if (dto instanceof NodeDefinitionDTO)
				nodeDefSet.add(((NodeDefinitionDTO) dto).getName());
			else if (dto instanceof WorkflowTemplateDTO)
				templateSet.add(((WorkflowTemplateDTO) dto).getName());
		}
		if (templateSet.size() > 0)
			message += "The node definition is being used by the following templates: "
					+ templateSet + "." + System.getProperty(LINE_SEPRATOR);
		if (nodeDefSet.size() > 0)
			message += "The node definition is being used by the following superstates: "
					+ nodeDefSet + "." + System.getProperty(LINE_SEPRATOR);
		message += "Are you sure you want to delete?";
		return message;
	}

	// ========================================================================
	// ===================== INTERNAL CLASSES FOR EVENT HANDLING
	// ========================================================================

	/*
	 * Handle the Selection changes for the Messages Table
	 */
	private class MessageSelectionChangedListener implements ISelectionChangedListener {
		/**
		 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
		 */
		public void selectionChanged(SelectionChangedEvent event) {
			ISelection selection = event.getSelection();
			IManagedForm managedForm = MessageMasterSectionPart.this.getManagedForm();

			MessageMasterSectionPart.this.updateButtonStatus();
			managedForm.fireSelectionChanged(MessageMasterSectionPart.this, selection);
		}
	}

	/*
	 * Handle the Button press for the control buttons
	 */
	private class ButtonPressListener extends SelectionAdapter {
		/**
		 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		public void widgetSelected(SelectionEvent event) {
			if (event.getSource() == MessageMasterSectionPart.this.createButton) {
				MessageMasterSectionPart.this.createNewMessage();
			} else if (event.getSource() == MessageMasterSectionPart.this.deleteButton) {
				MessageMasterSectionPart.this.deleteSelectedMessage();
			}
		}
	}

	// ========================================================================
	// ===================== INTERNAL CLASSES FOR TABLE VIEWER
	// ========================================================================

	/*
	 * The Label Provider for the Table
	 */
	private class MessageLabelProvider extends LabelProvider implements ITableLabelProvider {
		/**
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnImage(java.lang.Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}

		/**
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			if (!(element instanceof MessageInput))
				return null;

			MessageInput input = (MessageInput) element;
			Message message = input.getMessage();
			if (message == null)
				return null;

			String label = "";
			switch (columnIndex) {
			case MESSAGE_UID_INDEX:
				label = message.getUid();
				break;

			case MESSAGE_NAME_INDEX:
				label = message.getName();
				break;

			case MESSAGE_TIMEOUT_INDEX:
				if (nodesTimeout.get(message.getUid()) != null)
					label = nodesTimeout.get(message.getUid());
				break;
			}

			if (label != null)
				label = label.trim();

			return label;
		}

		/**
		 * @see org.eclipse.jface.viewers.LabelProvider#getText(java.lang.Object)
		 */
		@Override
		public String getText(Object element) {
			if (!(element instanceof MessageInput))
				return null;

			MessageInput input = (MessageInput) element;
			Message message = input.getMessage();
			if (message == null)
				return null;

			return message.getName();
		}
	}

	/*
	 * Content Provider for the Messages
	 */
	private class MessageContentProvider implements IStructuredContentProvider {
		/**
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		public Object[] getElements(Object inputElement) {
			Object[] elements = null;

			if (inputElement instanceof ResourceTypeEditorInput) {
				ResourceTypeEditorInput input = (ResourceTypeEditorInput) inputElement;
				ResourceType resourceType = input.getResourceType();
				boolean isEditable = input.canEdit();
				Set<Message> messages = (resourceType != null) ? resourceType.getMessages() : null;
				if (messages != null) {
					elements = new Object[messages.size()];
					int counter = 0;
					for (Message message : messages) {
						String timeout = nodesTimeout.get(message.getUid());
						MessageInput messageInput = new MessageInput(message, isEditable);
						messageInput.setTimeout(timeout);
						elements[counter++] = messageInput;
					}
				}
			} else {
				elements = new Object[0];
			}

			return elements;
		}

		/**
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer,
		 *      java.lang.Object, java.lang.Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) { /* Do nothing */
		}

		/**
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose() {
		}
	}
	
	protected void refreshViews() {
		IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
		IViewPart part = page.findView(ID_RESOURCE_TYPE);

		if (part != null)
			AbstractTreeViewExplorer.class.cast(part).refresh();
		
		part = page.findView(ProjectExplorer.VIEW_ID);
		
		if (part != null)
			((TreeViewer) part.getAdapter(CommonViewer.class)).refresh();
	}
}
