/**
 * NodeDefinitionComparer.java
 * Created Apr 15, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.compare;

import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.ADD;
import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.CHANGE;
import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.NONE;
import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.REMOVE;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITree;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITreeNode;
import com.tandbergtv.watchpoint.studio.ui.model.tree.provider.DefaultTree;
import com.tandbergtv.watchpoint.studio.ui.model.tree.provider.MutableTreeNode;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiffCalculator;
import com.tandbergtv.watchpoint.studio.ui.sync.SynchronizationException;
import com.tandbergtv.watchpoint.studio.ui.sync.impl.Diff;
import com.tandbergtv.watchpoint.studio.ui.sync.impl.UpdateElement;
import com.tandbergtv.watchpoint.studio.ui.util.NameValuePair;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

/**
 * Compares two node definitions, the "local" and its "remote" version
 * 
 * @author Sahil Verma
 */
public class NodeDefinitionComparer implements IDiffCalculator {
	
	protected NodeDefinitionDTO local;
	
	protected NodeDefinitionDTO remote;
	
	protected NodeDefinition localElement;
	
	protected NodeDefinition remoteElement;
	
	/**
	 * {@inheritDoc}
	 */
	public ITree<IDiff> compare(IWatchPointDTO dto1, IWatchPointDTO dto2, IProgressMonitor monitor) {
		this.local = NodeDefinitionDTO.class.cast(dto1);
		this.remote = NodeDefinitionDTO.class.cast(dto2);
		
		return compare(createSemanticElement(local), createSemanticElement(remote), monitor);
	}
	
	/**
	 * {@inheritDoc}
	 */
	public ITree<IDiff> compare(NodeDefinition local, NodeDefinition remote, IProgressMonitor monitor) {
		this.localElement = local;
		this.remoteElement = remote;
		
		checkRename();
		
		ITree<IDiff> tree = doCompare(monitor);
		
		return tree;
	}
	
	protected ITree<IDiff> doCompare(IProgressMonitor monitor) {
		ITree<IDiff> tree = createDiffTree();
		ITreeNode<IDiff> root = tree.getRoot();
		
		try {
			if (monitor != null) {
				monitor.beginTask("Calculating changes to variables", 3);
			}
			
			for (ITreeNode<IDiff> treenode : getAddedTreeNodes())
				root.addChild(treenode);
			
			incrementMonitor(monitor);
			
			for (ITreeNode<IDiff> treenode : getRemovedTreeNodes())
				root.addChild(treenode);
			
			incrementMonitor(monitor);
			
			for (ITreeNode<IDiff> treenode : getChangedTreeNodes())
				root.addChild(treenode);
			
			incrementMonitor(monitor);
			
			if (!root.hasChildren() && !hasChanged())
				return createEmptyTree(local);
		} finally {
			if (monitor != null) {
				monitor.done();
			}
		}
		
		return tree;
	}
	
	protected void incrementMonitor(IProgressMonitor monitor) {
		if (monitor != null) {
			monitor.worked(1);
		}
	}
	
	protected void checkRename() {
		if (!localElement.matchUids(remoteElement))
			throw new SynchronizationException("Node " + local.getName() + " - uid mismatch, current [" + 
				localElement.getUid() + "], new [" + remoteElement.getUid() + "]");
	}
	
	/**
	 * Gets the tree nodes containing diffs that represent variables that have been added to the node,
	 * including composite keys.
	 * 
	 * @return
	 */
	protected List<ITreeNode<IDiff>> getAddedTreeNodes() {
		List<ITreeNode<IDiff>> treenodes = new ArrayList<ITreeNode<IDiff>>();

		for (WPVariable remoteVariable : remoteElement.getVariables()) {
			if (!localElement.hasVariable(remoteVariable.getMappedName()))
				treenodes.add(new MutableTreeNode<IDiff>(new Diff(remoteVariable, ADD)));
		}

		return treenodes;
	}

	/**
	 * Gets the tree nodes containing diffs that represent variables that have been removed from the node,
	 * including composite keys.
	 * 
	 * @return
	 */
	protected List<ITreeNode<IDiff>> getRemovedTreeNodes() {
		List<ITreeNode<IDiff>> treenodes = new ArrayList<ITreeNode<IDiff>>();
		
		for (WPVariable localVariable : localElement.getVariables()) {
			if (!remoteElement.hasVariable(localVariable.getMappedName()))
				treenodes.add(new MutableTreeNode<IDiff>(new Diff(localVariable, REMOVE)));
		}
		
		return treenodes;
	}

	/**
	 * @return
	 */
	protected List<ITreeNode<IDiff>> getChangedTreeNodes() {
List<ITreeNode<IDiff>> treenodes = new ArrayList<ITreeNode<IDiff>>();
		
		for (WPVariable localVariable : localElement.getVariables()) {
			if (localVariable.isCompositeKey())
				continue;
			
			if (!remoteElement.hasVariable(localVariable.getMappedName()))
				continue;
			
			WPVariable remoteVariable = remoteElement.getVariable(localVariable.getMappedName());
			
			if (hasAccessChanged(localVariable, remoteVariable))
				treenodes.add(new MutableTreeNode<IDiff>(new Diff(remoteVariable, "access", CHANGE)));
			
			if (hasTypeChanged(localVariable, remoteVariable))
				treenodes.add(new MutableTreeNode<IDiff>(new Diff(remoteVariable, "type", CHANGE)));
		}
		
		return treenodes;
	}
	
	protected boolean hasAccessChanged(WPVariable local, WPVariable remote) {
		return !local.getAccess().equals(remote.getAccess());
	}
	
	protected boolean hasTypeChanged(WPVariable local, WPVariable remote) {
		String type1 = local.getEmptySafeType();
		String type2 = remote.getEmptySafeType();
		
		if (type1 == null && type2 == null)
			return false;
		
		if (type1 != null)
			return !type1.equals(type2);
		
		return true;
	}
	
	protected boolean hasChanged() {
		if (hasDueDateChanged())
			return true;
		
		/* Compare the constants, if any */
		List<NameValuePair> constants = localElement.getConstants();
		
		if (constants.size() != remoteElement.getConstants().size())
			return true;
		
		for (NameValuePair constant : constants) {
			String value = remoteElement.getConstantValue(constant.getName());
			
			if (value == null || !value.equals(constant.getValue()))
				return true;
		}
		
		return false;
	}
	
	protected boolean hasDueDateChanged() {
		if (localElement.getDueDate() == null && remoteElement.getDueDate() == null)
			return false;
		
		if (localElement.getDueDate() != null)
			return !localElement.getDueDate().equals(remoteElement.getDueDate());
		
		return true;
	}

	protected ITree<IDiff> createDiffTree() {
		IDiff diff = new UpdateElement(local, remote, remoteElement, CHANGE);
		ITree<IDiff> tree = new DefaultTree<IDiff>(new MutableTreeNode<IDiff>(diff));

		return tree;
	}
	
	protected ITree<IDiff> createEmptyTree(NodeDefinitionDTO local) {
		IDiff diff = new Diff(local, NONE);
		ITree<IDiff> tree = new DefaultTree<IDiff>(new MutableTreeNode<IDiff>(diff));

		return tree;
	}
	
	protected NodeDefinition createSemanticElement(NodeDefinitionDTO node) {
		// FIXME If this is a superstate, set IDs of children as well
		NodeDefinition element = null;

		try {
			element = SemanticElementUtil.createNodeDefinition(node.getXml());
			element.setId(node.getId());
			element.setResourceType(node.getResourceType().getId());
		} catch (Exception e) {
		}

		return element;
	}
}
