/**
 * DiffTreeVisitor.java
 * Created Apr 16, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.impl;

import static com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType.MessageNode;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;

import org.jbpm.gd.common.model.SemanticElement;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.dataaccess.WorkflowTemplateDTODAI;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.ui.command.AbstractCommand;
import com.tandbergtv.watchpoint.studio.ui.command.ICommand;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITreeNode;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITreeNodeVisitor;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.IUpdateElement;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.AddMessageCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.AddNodeCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.AddSuperStateCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.RemoveMessageCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.RemoveNodeCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.RemoveSuperStateCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.ResourceTypeCreateCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.ResourceTypeUpdateCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.UpdateMessageCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.UpdateNodeCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.resource.UpdateSuperStateCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.template.AddVariableCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.template.RemoveTemplateNodeCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.template.RemoveVariableCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.template.TemplateDiffCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.template.UpdateTemplateSuperStateCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.template.UpdateVariableCommand;
import com.tandbergtv.watchpoint.studio.ui.sync.util.ISynchronizationContext;
import com.tandbergtv.watchpoint.studio.ui.sync.util.SynchronizationContext;

/**
 * Visits the diff tree of a resource type and builds a list of commands that need to be executed
 * in order to apply each diff
 * 
 * @author Sahil Verma
 */
public class DiffTreeVisitor implements ITreeNodeVisitor<IDiff> {
	
	private final ResourceType type;
	
	/**
	 * The list of commands. Note that a re-ordered version of this list is returned to callers.
	 */
	private List<ICommand> commands;
	
	protected final ISynchronizationContext context;
	
	/**
	 * Priorities for each type of command, highest on top
	 */
	@SuppressWarnings("serial")
	private static final Map<Class<? extends ICommand>, Integer> PRIORITIES = 
		new HashMap<Class<? extends ICommand>, Integer>() {{
		put(AddMessageCommand.class, 0);
		put(AddNodeCommand.class, 1);
		put(AddSuperStateCommand.class, 2);
		put(UpdateMessageCommand.class, 3);
		put(UpdateNodeCommand.class, 4);
		put(UpdateSuperStateCommand.class, 5);
		put(TemplateDiffCommand.class, 6);
		put(RemoveSuperStateCommand.class, 7);
		put(RemoveNodeCommand.class, 8);
		put(RemoveMessageCommand.class, 9);
		put(ResourceTypeUpdateCommand.class, 10);
		put(ResourceTypeCreateCommand.class, 11);
	}};
	
	/**
	 * Creates a visitor
	 * 
	 * @param type the resource type that is undergoing modification
	 * @param context sync context to aid persistence operations
	 */
	public DiffTreeVisitor(ResourceType type, ISynchronizationContext context) {
		this.type = type;
		this.context = context;
		this.commands = new ArrayList<ICommand>();
	}

	/**
	 * Returns the ordered list of commands. Callers should call visit() before calling this method.
	 * 
	 * @return the commands
	 */
	public Collection<ICommand> getCommands() {
		return getOrderedCommands();
	}

	/**
	 * {@inheritDoc}
	 */
	public void visit(ITreeNode<IDiff> node) {
		IDiff diff = node.getData();
		
		switch (diff.getKind()) {
		case NONE:
			break;
		case ADD:
			added(node);
			break;
		case REMOVE:
			removed(node);
			break;
		case CHANGE:
			changed(node);
			break;
		}
	}
	
	/**
	 * Adds one or more commands due to the given tree node that wraps a diff of type ADD
	 * @param node
	 */
	protected void added(ITreeNode<IDiff> node) {
		IDiff diff = node.getData();
		SemanticElement element = diff.getElement();
		
		if (diff.getModel() instanceof ResourceType)
			resourceTypeAdded(diff);
		if (diff.getModel() instanceof Message)
			messageAdded(diff);
		else if (element instanceof NodeDefinition)
			nodeAdded(diff);
		else if (element instanceof Variable)
			variableAdded(diff, node.getParent().getData());
	}
	
	/**
	 * Adds one or more commands due to the given tree node that wraps a diff of type REMOVE
	 * 
	 * @param node
	 */
	protected void removed(ITreeNode<IDiff> node) {
		IDiff diff = node.getData();
		SemanticElement element = diff.getElement();
		
		if (diff.getModel() instanceof Message)
			messageRemoved(diff);
		else if (element instanceof NodeDefinition)
			nodeRemoved(diff);
		else if (element instanceof Variable)
			variableRemoved(diff, node.getParent().getData());
	}
	
	/**
	 * Adds one or more commands due to the given tree node that wraps a diff of type CHANGE
	 * 
	 * @param node
	 */
	protected void changed(ITreeNode<IDiff> node) {
		IDiff diff = node.getData();
		SemanticElement element = diff.getElement();
		
		if (diff.getModel() instanceof ResourceType)
			resourceTypeChanged(diff);
		else if (diff.getModel() instanceof Message)
			messageChanged(diff);
		else if (element instanceof NodeDefinition)
			nodeChanged(diff);
		else if (element instanceof Variable)
			variableChanged(diff, node.getParent().getData());
	}
	
	protected void resourceTypeAdded(IDiff diff) {
		addCommand(new ResourceTypeCreateCommand("resourcetype-create", diff, context));
	}
	
	protected void resourceTypeChanged(IDiff diff) {
		addCommand(new ResourceTypeUpdateCommand("resourcetype-update", diff, context));
	}
	
	/**
	 * Generates a command based on the diff which represents a new message
	 * @param diff
	 */
	protected void messageAdded(IDiff diff) {
		addCommand(new AddMessageCommand("message-add", diff, context));
	}
	
	/**
	 * Generates a command based on the diff which represents an updated message
	 * @param diff
	 */
	protected void messageChanged(IDiff diff) {
		addCommand(new UpdateMessageCommand("message-update", diff, context));
	}
	
	/**
	 * Generates a command based on the diff which represents a deleted message
	 * 
	 * @param diff
	 */
	protected void messageRemoved(IDiff diff) {
		addCommand(new RemoveMessageCommand("message-delete", diff, context));
	}
	
	protected void nodeAdded(IDiff diff) {
		NodeDefinitionDTO node = (NodeDefinitionDTO) diff.getModel();
		
		if (node.getType() == MessageNode)
			addCommand(new AddNodeCommand("node-add", diff, context));
		else
			addCommand(new AddSuperStateCommand("superstate-add", diff, context));
	}
	
	/**
	 * Generates one or more commands based on an update to a node. There is one command for the
	 * update to the node, as well as one each for every template that uses the node.
	 * 
	 * @param diff
	 */
	protected void nodeChanged(IDiff diff) {
		IUpdateElement update = IUpdateElement.class.cast(diff);
		NodeDefinitionDTO dto = (NodeDefinitionDTO) update.getLocalModel();
		NodeDefinitionDTO node = (NodeDefinitionDTO) update.getModel();
		
		node.setId(dto.getId());
		
		if (dto.getType() == MessageNode) {
			addCommand(new UpdateNodeCommand("node-update", diff, context));
		} else {
			addCommand(new UpdateSuperStateCommand("superstate-update", diff, context));
		
			if (isUsedInTemplates(dto)) {
				for (WorkflowTemplateDTO template : getTemplates(dto)) {
					TemplateDiffCommand composite = getTemplateDiffCommand(template);
					UpdateTemplateSuperStateCommand command = 
						new UpdateTemplateSuperStateCommand("superstate-template-update", diff, context);
					
					command.setModel(composite.getModel());
//					command.setContainer(composite.getNodeElementContainer());
					
					composite.addCommand(command);
				}
			}
		}
	}
	
	/**
	 * Generates one or more commands based on the diff which represents a deleted node. One command
	 * represents the actual node removal. Additionally, one command is generated per template that
	 * uses the removed node. If the command corresponding to the template is already in our list, it
	 * is used as is. Otherwise a new composite command is created.
	 * 
	 * @param diff
	 */
	protected void nodeRemoved(IDiff diff) {
		NodeDefinitionDTO dto = (NodeDefinitionDTO) diff.getModel();
		
		if (isUsedInTemplates(dto)) {
			for (WorkflowTemplateDTO template : getTemplates(dto)) {
				TemplateDiffCommand composite = getTemplateDiffCommand(template);
				
				doNodeRemoved(diff, composite);
			}
		}
	
		if (dto.getType() == MessageNode)
			addCommand(new RemoveNodeCommand("node-delete", diff, context));
		else
			addCommand(new RemoveSuperStateCommand("superstate-delete", diff, context));
	}
	
	protected void doNodeRemoved(IDiff diff, TemplateDiffCommand composite) {
		RemoveTemplateNodeCommand command = new RemoveTemplateNodeCommand("template-node-delete", diff, context);
		
		command.setModel(composite.getModel());
		command.setNodeElementContainer(composite.getNodeElementContainer());
		
		composite.addCommand(command);
	}
	
	/**
	 * Generates one or more commands based on the diff which represents a new variable in a node.
	 * 
	 * @param diff
	 * @param parent
	 */
	protected void variableAdded(IDiff diff, IDiff parent) {
		String name = ((NodeDefinitionDTO) parent.getModel()).getName();
		NodeDefinitionDTO dto = getExistingNodeDefinition(name);
		NodeDefinition node = (NodeDefinition) parent.getElement();
		
		if (isUsedInTemplates(dto)) {
			for (WorkflowTemplateDTO template : getTemplates(dto)) {
				TemplateDiffCommand composite = getTemplateDiffCommand(template);
				AddVariableCommand command = new AddVariableCommand("variable-add", diff, context);
				
				command.setModel(composite.getModel());
				command.setNodeElementContainer(composite.getNodeElementContainer());
				command.setNode(node);
				
				composite.addCommand(command);
			}
		}
	}
	
	protected void variableChanged(IDiff diff, IDiff parent) {
		String name = ((NodeDefinitionDTO) parent.getModel()).getName();
		NodeDefinitionDTO dto = getExistingNodeDefinition(name);
		NodeDefinition node = (NodeDefinition) parent.getElement();
		
		if (isUsedInTemplates(dto)) {
			for (WorkflowTemplateDTO template : getTemplates(dto)) {
				TemplateDiffCommand composite = getTemplateDiffCommand(template);
				UpdateVariableCommand command = new UpdateVariableCommand("variable-update", diff, context);
				
				command.setModel(composite.getModel());
				command.setNodeElementContainer(composite.getNodeElementContainer());
				command.setNode(node);
				
				composite.addCommand(command);
			}
		}
	}
	
	protected void variableRemoved(IDiff diff, IDiff parent) {
		String name = ((NodeDefinitionDTO) parent.getModel()).getName();
		NodeDefinitionDTO dto = getExistingNodeDefinition(name);
		
		if (isUsedInTemplates(dto)) {
			for (WorkflowTemplateDTO template : getTemplates(dto)) {
				TemplateDiffCommand composite = getTemplateDiffCommand(template);
				RemoveVariableCommand command = new RemoveVariableCommand("variable-delete", diff, context);
				
				command.setModel(composite.getModel());
				command.setNodeElementContainer(composite.getNodeElementContainer());
				command.setNode(dto);
				
				composite.addCommand(command);
			}
		}
	}
	
	/**
	 * Gets a command corresponding to the given template
	 * 
	 * @param template
	 * @return
	 */
	protected TemplateDiffCommand getTemplateDiffCommand(WorkflowTemplateDTO template) {
		/* Lookup the list to check if we've already got a command for our template */
		for (ICommand command : this.commands) {
			if (!(command instanceof TemplateDiffCommand))
				continue;
			
			TemplateDiffCommand cmd = TemplateDiffCommand.class.cast(command);
			
			if (cmd.getModel().equals(template))
				return cmd;
		}
		
		/* Create a new command and add it to the beginning of the list */
		TemplateDiffCommand	cmd = new TemplateDiffCommand("template-update", "Template updated", context);
		
		cmd.setModel(template);
		addCommand(cmd);
		
		return cmd;
	}
	
	protected NodeDefinitionDTO getExistingNodeDefinition(String name) {
		return this.type.getNode(name);
	}
	
	protected Collection<WorkflowTemplateDTO> getTemplates(NodeDefinitionDTO node) {
		/* FIXME Optimize - fetch templates for a node definition once and cache them */
		//return getTemplateDAO().findByNodeDefinitionUsage(node.getId(), false);
		return Collections.emptyList();
	}
	
	protected boolean isUsedInTemplates(NodeDefinitionDTO node) {
		//return getTemplateDAO().getCountByNodeDefinitionUsage(node.getId(), false) > 0;
		return false;
	}
	
	protected WorkflowTemplateDTODAI getTemplateDAO() {
		return SynchronizationContext.class.cast(context).getTemplateDAO();
	}
	
	protected void addCommand(ICommand command) {
		this.commands.add(command);
	}
	
	/**
	 * Order the commands by their priority
	 * 
	 * @return
	 */
	protected List<ICommand> getOrderedCommands() {
		List<ICommand> list = new ArrayList<ICommand>();
		int size = commands.size();
		
		PriorityQueue<OrderedCommand> queue = new PriorityQueue<OrderedCommand>(size, new Comparator<OrderedCommand>() {
			public int compare(OrderedCommand o1, OrderedCommand o2) {
				return o1.priority - o2.priority;
			}
		});

		for (ICommand command : this.commands)
			enqueueCommand(queue, command);
		
		while (!queue.isEmpty())
			list.add(queue.poll());
		
		return list;
	}
	
	protected void enqueueCommand(PriorityQueue<OrderedCommand> queue, ICommand command) {
		int priority = getPriority(command.getClass());
		
		queue.offer(new OrderedCommand(command, priority));
	}
	
	protected int getPriority(Class<? extends ICommand> clazz) {
		return PRIORITIES.get(clazz);
	}

	/**
	 * Decorates a command with a priority
	 */
	class OrderedCommand extends AbstractCommand {
		int priority;
		
		ICommand delegate;
		
		OrderedCommand(ICommand delegate, int priority) {
			super(delegate.getName(), delegate.getDescription());
			this.delegate = delegate;
			this.priority = priority;
		}

		public void execute() {
			delegate.execute();
		}

		public String toString() {
			return delegate.toString();
		}
	}
}
