package com.tandbergtv.watchpoint.studio.ui.view.resourcetype;

import java.util.ArrayList;
import java.util.List;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.ui.statushandlers.StatusManager;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.view.DefaultContentProvider;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;

/**
 * The Resource Type Content Provider used to fill the menu.
 * @author Patrik Araujo 
 */
public class ResourceTypeProjectExplorerContentProvider extends DefaultContentProvider {
	private static final String constantsXpath = "//map";
	private static final String compositeKeysXpath = "//keys";
	private static final String variablesXpath = "//variable";

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object element) {
		Object[] result = null;
		
		if (element instanceof IProject) {
			IProject project = (IProject) element;
			IResourceTypeService service = ServiceFactory.createFactory().createResourceTypeService();
			String projectName = project.getName();
			
			// Try to get the resource type already builded
			List<ResourceType> resourceTypeList = service.getResourceTypeByProject(projectName);
			if (!resourceTypeList.isEmpty()) {
				result = new Object[1];
				result[0] = resourceTypeList.get(0);
			}
		}
		
		// Lists the nodes for a given resource type
		if(element instanceof ResourceType){
			List<NodeDefinitionDTO> nodeDefinitions = new ArrayList<NodeDefinitionDTO>();
			IResourceTypeService service = ServiceFactory.createFactory().createResourceTypeService();
			
			// Always gets the resource type from database in order to get the updated object
			ResourceType resourceType = service.getResourceTypeByProject( ((ResourceType)element).getProjectName() ).get(0);
			nodeDefinitions.addAll(resourceType.getNodes());
			
			result = nodeDefinitions.toArray();
		}
		
		// Lists the parameters for a given node definition
		if(element instanceof NodeDefinitionDTO){
			NodeDefinitionDTO nodeDTO = (NodeDefinitionDTO) element;
			INodeDefinitionService service = ServiceFactory.createFactory().createNodeDefinitionService();
			
			// Always gets the node definition from database in order to get the updated object
			NodeDefinitionDTO persisted = service.getNodeDefinitionDTOByPath(nodeDTO.getPath());
			List<Object> vars = new ArrayList<Object>();
			try {
				NodeDefinition nodeDef = SemanticElementUtil.createNodeDefinition(persisted.getXml());
				vars.addAll(nodeDef.getVariables());
				vars.addAll(nodeDef.getConstants());
            } catch (Exception e) {
				handleException("Error creating Node Definition from NodeDefinitionDTO xml", e);
			}
			
			if (vars != null && vars.size() > 0)
				result = vars.toArray();
		}
		
		return result;
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
	 */
	public Object getParent(Object element) {
		/* Returns null because this method is being called for the project object 
		   and the project has no parent. */ 
		return null;
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object element) {
		boolean result = false;
		// Check if the resource type has nodes
		if(element instanceof ResourceType){
			ResourceType resourceType = (ResourceType)element;
			if(resourceType.getNodes().size() > 0){
				return true;
			}
		}
		// Check if the node definition has variables, constants or composite keys
		if (element instanceof NodeDefinitionDTO) {
			NodeDefinitionDTO node = (NodeDefinitionDTO) element;
			// Checking using xpath is more efficient than parsing the node definition to a 
			// node definition object and check the variables list.
			try {
				Document doc = XMLDocumentUtility.loadXml(node.getXml());
				
				XPath xpath = XPathFactory.newInstance().newXPath();
	
				NodeList constants = (NodeList)xpath.evaluate(constantsXpath, doc.getDocumentElement(), XPathConstants.NODESET);
				NodeList compositeKeys = (NodeList)xpath.evaluate(compositeKeysXpath, doc.getDocumentElement(), XPathConstants.NODESET);
				NodeList variables = (NodeList)xpath.evaluate(variablesXpath, doc.getDocumentElement(), XPathConstants.NODESET);
				
				if(constants.getLength() > 0 || compositeKeys.getLength() > 0 || variables.getLength() > 0){
					return true;
				}
			} catch (Exception e) {
				handleException("Error evaluating if the Node Definition " + node.getName() + " has childen.", e);
			}
		}
		return result;
	}
	
	protected void handleException(String errorMessage, Exception e){
		Status status = new Status(IStatus.ERROR, "WatchPoint Studio",IStatus.ERROR, errorMessage, e);
		StatusManager.getManager().handle(status, StatusManager.SHOW);
	}
	
}