package com.tandbergtv.watchpoint.studio.ui.view.resourcetype.actions;

import static com.tandbergtv.watchpoint.studio.ui.wizard.RTPConstants.ADAPTOR_FOLDER_PATH;
import static org.eclipse.swt.widgets.Display.getCurrent;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.window.Window;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.jbpm.gd.jpdl.model.NodeElement;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.util.ServiceErrors;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.ResourceTypeMessageDeleteDialog;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

public class DeleteMessageAction extends AbstractNodeDefinitionAction {
	
	private NodeDefinitionDTO dto;
	ResourceTypeMessageDeleteDialog dialog;
	
	@Override
	public void run(IAction action) {
		NodeDefinition nodeDefParsed = null;
		try {
			nodeDefParsed = SemanticElementUtil.createNodeDefinition(dto.getXml());
			nodeDefParsed.setId(dto.getId());
        } catch (Exception e) {
			Utility.handleException("Failed creating node definition from xml.", e);
		}
		
		/* get the references for the node if the node is being used by templates or superstates*/
		this.nodeReferences = checkNodeInUse(nodeDefParsed);
		dialog = buildDialog(nodeDefParsed);
		if (dialog.open() != Window.OK)
			return;
		
		deleteMessage(nodeDefParsed);
		
		/* Check if the node definition is being used in a superstate. If true, updates the
		   superstate */
		for (IWatchPointDTO referenceDTO : this.nodeReferences) {
			if(referenceDTO instanceof NodeDefinitionDTO){
				NodeDefinitionDTO nodeDTO = (NodeDefinitionDTO) referenceDTO;
				if(nodeDTO.getType().equals(NodeDefinitionType.SuperState)){
					removeNodeDefinitionInSuperstate(nodeDTO, nodeDefParsed, dto.getName());
				}
			}
		}
		
		if(dialog.getUpdateReferences()){
			try {
				new TemplateNodesRefactoring().deleteMessageNode(dialog.getAffectedMessages(), dto.getName());
			} catch (Exception e) {
				handleException("Error refactoring the message nodes", e);
			} 
		}
		
		getTreeViewer().refresh(dto.getResourceType());
	}
	
	private void removeNodeDefinitionInSuperstate(NodeDefinitionDTO nodeDTO, NodeDefinition node, String nodeName) {
		NodeDefinition superstateNodeDefinition = null;
		try {
			superstateNodeDefinition = SemanticElementUtil.createNodeDefinition(nodeDTO.getXml());
        } catch (Exception e) {
			handleException("Error creating message node from xml: " + nodeDTO.getName(), e);
			return;
		}
		
		XmlAdapter adapter = Utility.getAdapterFromString(nodeDTO.getXml(), node.getFactory());
		adapter.initialize(superstateNodeDefinition);
		
		NodeGroup nodeGroup = (NodeGroup)superstateNodeDefinition.getNode();
		NodeDefinition nodeDefinitionInSuperstate = null;
		for (NodeElement nodeElement : nodeGroup.getNodeElements()) {
			if(nodeElement instanceof NodeDefinition && ((NodeDefinition) nodeElement).getDefinitionName().equals(nodeName) ){
				nodeDefinitionInSuperstate = (NodeDefinition)nodeElement;
			}
		}
		superstateNodeDefinition.removeNodeElement(nodeDefinitionInSuperstate);
		
		String xml = Utility.getXMLFromDocument(adapter.getNode().getOwnerDocument());
		nodeDTO.setXml(xml);
		
		INodeDefinitionService service = ServiceFactory.createFactory().createNodeDefinitionService();
		service.updateNodeDefinitionInFileSystem(nodeDTO);
	}

	private void deleteMessage(NodeDefinition nodeDefinition){
		boolean failedRemovingNode = false;
		ResourceType resourceType = dto.getResourceType();
		
		INodeDefinitionService nodeDefinitionService = null;
		try {
			nodeDefinitionService = ServiceFactory.createFactory().createNodeDefinitionService();
			nodeDefinitionService.deleteNodeDefinitionInFileSystem(dto, true);
		} 
		catch (ServiceException e) {							
			failedRemovingNode = true;
			handleException(ServiceErrors.getInstance().getProperty(e.getServiceErrorCode().getCode()), e);
			return;//to skid delete message when locked?
		} 

		if (!failedRemovingNode) {
			Message message = resourceType.getMessage(nodeDefinition.getUid());
			resourceType.removeMessage(message);

			// Delete adaptor definition for the message
			IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(resourceType.determineNameForProject());
			
			if (project.exists()) {
				try {
					project.getFolder(ADAPTOR_FOLDER_PATH + message.getUid()).delete(true, true, null);
				} catch (CoreException e) {
					handleException("Message deleted but unable to delete message folder from the resource type project.", e);
				}
			}
			
			ServiceFactory factory = ServiceFactory.createFactory();
			IResourceTypeService resourceTypeService = factory.createResourceTypeService();
			
			resourceTypeService.updateResourceTypeInFileSystem(resourceType);

			ResourceType updated = resourceTypeService.getResourceTypeFromFileSystem(resourceType.getProjectName());
			
			if (updated == null) {
				Utility.reportError("Error updating resource type", null);
			} 
		}
	}
	
	protected ResourceTypeMessageDeleteDialog buildDialog(NodeDefinition node) {
		ResourceTypeMessageDeleteDialog dialog = 
			new ResourceTypeMessageDeleteDialog(getCurrent().getActiveShell());
		
		dialog.initReferences( buildReferencesMap(dto) );
		dialog.setMessageName(dto.getName());
		
		dialog.setBlockOnOpen(true);
		return dialog;
	}
	
	@Override
	public void selectionChanged(IAction action, ISelection selection) {
		if (!selection.isEmpty()) {
			Object element = ((TreeSelection) selection).getFirstElement();
			if (element instanceof NodeDefinitionDTO) {
				this.dto = (NodeDefinitionDTO) element;
			}
		}
	}

}
