package com.tandbergtv.watchpoint.studio.ui.view.resourcetype.actions;

import static org.eclipse.swt.widgets.Display.getCurrent;

import java.lang.reflect.InvocationTargetException;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.TreePath;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.jbpm.gd.jpdl.model.Description;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.service.INodeDefinitionService;
import com.tandbergtv.watchpoint.studio.service.ServiceException;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.NameValuePair;
import com.tandbergtv.watchpoint.studio.ui.util.ServiceErrors;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.ResourceTypeMessageDialog;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

public class ModifyParameterAction extends AbstractNodeDefinitionAction {

	private WPVariable variable;
	private NameValuePair constant;
	private NodeDefinitionDTO nodeDefinitionDTO;
	protected ResourceTypeMessageDialog dialog;
	protected String varoldName;

	@Override
	public void run(IAction action) {
		run();
	}
	
	public void run(WPVariable variable, NodeDefinitionDTO nodeDefinitionDTO) {
		this.variable = variable;
		this.nodeDefinitionDTO = nodeDefinitionDTO;
		run();
	}
	
	public void run(NameValuePair constant, NodeDefinitionDTO nodeDefinitionDTO) {
		this.constant = constant;
		this.nodeDefinitionDTO = nodeDefinitionDTO;
		run();
	}
	
	protected ResourceTypeMessageDialog buildDialog(NodeDefinition node) {
		ResourceTypeMessageDialog dialog = new ResourceTypeMessageDialog(getCurrent().getActiveShell());
		
		dialog.initReferences( buildReferencesMap(nodeDefinitionDTO) );
		dialog.setMessageName(nodeDefinitionDTO.getName());
		
		NameValuePair nvvar = null;

		// fill values into the dialog
		if (variable != null) {
			WPVariable wpvar = variable;
			varoldName = wpvar.getMappedName();

			dialog.setName(varoldName);
			dialog.setDatatype(wpvar.getType());
			dialog.setOutgoing(wpvar.getAccess() != null && wpvar.getAccess().indexOf("read") > -1);
			String compositeKey = node.getCompositeKey(varoldName);
			if (compositeKey != null && compositeKey.length() > 0) {
				dialog.setComposite(true);
				//set the properties on the dialog to modify a composite key
				dialog.setModifyCompKeyMode(true);
			}
			dialog.setConstant(wpvar.getValue());
			//check if the node is a superstate node
			if(node.getNodeType() == NodeDefinitionType.SuperState) {
				dialog.setSuperstate(true);
			}
			//set the properties on the dialog to modify a variable
			dialog.setModifyVariableMode(true);
		}

		if (constant != null) {
			nvvar = constant;
			varoldName = nvvar.getName();
			dialog.setName(nvvar.getName());
			dialog.setConstant(nvvar.getValue());
			//set the properties on the dialog to modify a constant
			dialog.setModifyConstantMode(true);
		}
		
		NodeDefinition cloneNode = node;
		Description oldDescription = cloneNode.getDescription();
		String olddesc = (oldDescription != null) ? oldDescription.getDescription() : "";
		String matchVar = PREFIX_PARAM + varoldName + " :";
		int inx = olddesc.indexOf(matchVar);
		
		if (inx >= 0 && olddesc.length() > (inx+matchVar.length())) {
			String subdesc = olddesc.substring(inx + matchVar.length() + 1 );
			int subStringEndIndex = getEndIndex(subdesc);
			olddesc = subdesc.substring(0, (subStringEndIndex > 0) ? subStringEndIndex : subdesc.length());
			dialog.setDescription(olddesc.trim());
		} else { 
			dialog.setDescription("");
		}
		dialog.setBlockOnOpen(true);

		return dialog;
	}

	@Override
	protected void updateNodeInSuperstate(NodeDefinition nodeSuperstate) {
		modifyParameter(nodeSuperstate);
	}
	
	protected void modifyParameter(NodeDefinition nodeDef) {

		String name = dialog.getName();
		if (name == null || name.length() == 0) {
			MessageDialog.openError(Display.getCurrent()
					.getActiveShell(), TITLE_ERROR,
					"Variable name can not be empty.");
			return;
		}
		
		if(name.contains(":")){
			MessageDialog.openError(Display.getCurrent()
					.getActiveShell(), TITLE_ERROR,
					"Variable name can not contain ':' character.");
			return;
		}
		
		String cons = dialog.getConstant();
		if (dialog.getConsButtonSelection() && (cons == null || cons.length() == 0)) {
			MessageDialog.openError(
					Display.getCurrent().getActiveShell(), TITLE_ERROR, "Constant value can not be empty.");
			return;
		}
		
		SemanticElementFactory factory = nodeDef.getFactory();
		WPVariable var = (factory == null) ? null
				: (WPVariable) factory.createById(SemanticElementConstants.VARIABLE_SEID);

		if (var == null) {
			MessageDialog.openError(Display.getCurrent().getActiveShell(), TITLE_ERROR,
					"Modification ended since no factory can be found for the node definition.");
			return;
		}
		var.setMappedName(name);
		String datatype = dialog.getDatatype();
		var.setType(datatype);
		
		if (cons != null && !cons.isEmpty()) {
			var.setValue(cons);
		}
		if (dialog.isOutgoing()) {
			var.setAccess("read");
		} else {
			var.setAccess("write");
		}

		setupConstant(nodeDef, name, var);

		setupDescription(nodeDef, name);
	}

	private void setupConstant(NodeDefinition nodeDef, String name, WPVariable var) {
		// Creates a constant object if its necessary
		NameValuePair constantValue = null;
		if (dialog.getConsButtonSelection()) {
			constantValue = new NameValuePair();
			constantValue.setName(dialog.getName());
			constantValue.setValue(dialog.getConstant());
		}

		if (this.constant == null) {
			// Removes the composite key and the variable.
			// This is useful in case of change from composite key for variable
			// and vice versa
			((IMessageContainer) nodeDef.getNode()).removeCompositeKey(varoldName);
			((IMessageContainer) nodeDef.getNode()).removeVariable(varoldName);

			// in case of composite key
			if (dialog.isComposite()) {
				nodeDef.setCompositeKey(name, name);
			} else {
				// in case of making a variable constant
				if (constantValue != null) {
					nodeDef.setConstantValue(varoldName, null);
					nodeDef.setConstantValue(constantValue.getName(), constantValue.getValue());
				} else {
					// in case of variable
					if (nodeDef.getNodeType() == NodeDefinitionType.SuperState) {
						nodeDef.removeVariable(varoldName);
						nodeDef.addVariable(var);
					} else {
						((IMessageContainer) nodeDef.getNode()).addVariable(var);
					}
				}
			}
		} else if (constantValue != null) {
			nodeDef.setConstantValue(varoldName, null);
			nodeDef.setConstantValue(constantValue.getName(), constantValue.getValue());
		}
	}

	private void setupDescription(NodeDefinition nodeDef, String name) {
		String varDesc = dialog.getDescription();
		Description description = nodeDef.getDescription();
		String desc = (description != null) ? description.getDescription() : null;

		// always add "@param varname : desc"
		if (description == null || "".equals(description.getDescription())) {
			setNewNodeDefDescription(nodeDef, PREFIX_PARAM + name + " : " + varDesc);
		} else {
			// clear the variable (set to null)
			description.setDescription(parseDescription(desc, varoldName, null));
			// set the value
			if (description.getDescription() == null || "".equals(description.getDescription())) {
				setNewNodeDefDescription(nodeDef, PREFIX_PARAM + name + " : " + varDesc);
			} else {
				description.setDescription(parseDescription(description.getDescription(), name, varDesc));
			}
		}
	}

	protected void run() {
		if (nodeDefinitionDTO == null) {
			return;
		}
			
		NodeDefinition nodeDefParsed = null;
		try {
			nodeDefParsed = SemanticElementUtil.createNodeDefinition(nodeDefinitionDTO.getXml());
			nodeDefParsed.setId(nodeDefinitionDTO.getId());
        } catch (Exception e) {
			handleException("Error creating Node Definition from NodeDefinitionDTO xml", e);
        }
		
		/* get the references for the node if the node is being used by templates or superstates*/
		this.nodeReferences = checkNodeInUse(nodeDefParsed);
		
		dialog = buildDialog(nodeDefParsed);
			
		if (dialog.open() != Window.OK)
			return;
			
		Object nodeele = nodeDefParsed;
		NodeDefinition nodeDef = (nodeele != null)? ((NodeDefinition)nodeele):null;
		try {
			
			XmlAdapter adapter = Utility.getAdapterFromString(nodeDefinitionDTO
					.getXml(), nodeDef.getFactory());
			adapter.initialize(nodeDef);
			
			modifyParameter(nodeDef);
			
			/* Check if the node definition is being used in a superstate. If true, updates the
			   superstate */
			for (IWatchPointDTO referenceDTO : this.nodeReferences) {
				if(referenceDTO instanceof NodeDefinitionDTO){
					NodeDefinitionDTO nodeDTO = (NodeDefinitionDTO) referenceDTO;
					if(nodeDTO.getType().equals(NodeDefinitionType.SuperState)){
						updateSuperstateNodeDefinition(nodeDTO, nodeDef, nodeDefinitionDTO.getName());
					}
				}
			}
			
			String xml = Utility.getXMLFromDocument(adapter.getNode()
					.getOwnerDocument());

			nodeDefinitionDTO.setXml(xml);
			
			INodeDefinitionService service = ServiceFactory.createFactory().createNodeDefinitionService();
			service.updateNodeDefinitionInFileSystem(nodeDefinitionDTO);
			
			if(dialog.getUpdateReferences()){
				try {
					new TemplateNodesRefactoring().refactorMessageNodes(dialog.getAffectedMessages(), nodeDefinitionDTO, nodeDefinitionDTO.getName());
				} catch (InvocationTargetException e) {
					handleException("Error refactoring the message nodes", e);
				} catch (InterruptedException e) {
					handleException("Error refactoring the message nodes", e);
				}
			}
		} catch (ServiceException e) {
			Utility.handleException(ServiceErrors.getInstance()
					.getProperty(e.getServiceErrorCode().getCode()), e);
		}
	}

	@Override
	public void selectionChanged(IAction action, ISelection selection) {
		if(!selection.isEmpty()){
			// Gets the node definition
			TreePath[] paths = ((TreeSelection) selection).getPaths();
			TreePath path = (paths != null) ? paths[0] : null;
			if(path != null){
				this.nodeDefinitionDTO = (NodeDefinitionDTO) path.getSegment(2);
			}
			// Gets the WPVariable
			Object element = ((TreeSelection) selection).getFirstElement();
			if (element instanceof WPVariable) {
				this.variable = (WPVariable) element;
			}
			// Gets the name value pair
			if (element instanceof NameValuePair) {
				this.constant = (NameValuePair) element;
			}
		}
	}

}
