package com.tandbergtv.watchpoint.studio.ui.wizard.resourcegroup;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.wizards.newresource.BasicNewResourceWizard;

import com.tandbergtv.watchpoint.studio.dto.ResourceGroup;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.util.FileUtil;

/**
 * 		Wizard for new ResourceGroup creation
 * 
 * @author <a href="francisco.neto@venturus.org.br">vntfrne Francisco Bento da Silva Neto</a>
 *
 */
public class NewResourceGroupWizard extends Wizard implements INewWizard {

	// The Logger
	private static final Logger logger = Logger.getLogger(NewResourceGroupWizard.class);

	private static final String RESOURCE_GROUP_NAME_STRING = "resourceGroupNameToReplace";
	private static final String SYSTEM_ID_STRING = "systemIdToReplace";
	
	private IStructuredSelection selection;
	private NewResourceGroupWizardPage page;
	private String suggestedFolder;
	
	public NewResourceGroupWizard() {
		setWindowTitle("New Resource Group");
	}
	public NewResourceGroupWizard(String suggestedFolder) {
		this();
		this.suggestedFolder = suggestedFolder;
	}
	

	public void init(IWorkbench workbench, IStructuredSelection currentSelection) {
		this.selection= currentSelection;
	}
	
	public void addPages() {
		page = new NewResourceGroupWizardPage(suggestedFolder);
		addPage(page);
		page.init(selection);
	}
	
	public boolean performFinish() {
		try {
			IFolder folder = page.getProcessFolder();
			
			// If it is creating inside a ResourceType project, automatically sets the ResourceType systemId in the Group. 
			IResourceTypeService rtService = ServiceFactory.createFactory().createResourceTypeService();
			String systemId = null;
			List<ResourceType> resourceTypes = rtService.getResourceTypeByProject(folder.getProject().getName());
			if (resourceTypes.size() == 1) {
				ResourceType rt = resourceTypes.get(0);
				systemId = rt.getSystemId();
			}
			
			IFile resourceGroupFile = createResourceGroup(folder, page.getProcessName(), systemId);
			IDE.openEditor(getActivePage(), resourceGroupFile);
			BasicNewResourceWizard.selectAndReveal(resourceGroupFile, getActiveWorkbenchWindow());
			return true;
		} catch (CoreException e) {
			e.printStackTrace();
			return false;
		}
	}

	private IWorkbenchPage getActivePage() {
		return getActiveWorkbenchWindow().getActivePage();
	}

	private IWorkbenchWindow getActiveWorkbenchWindow() {
		return PlatformUI.getWorkbench().getActiveWorkbenchWindow();
	}
	
	/**
	 * Creates a ResourceGroup in the Eclipse's workspace.
	 * Group will be placed on folder/groupName 
	 * 
	 * @param folder The root folder for the group. eg.: MyTemplate/groups 
	 * @param groupName The groupName
	 * @param systemId (optional) The systemId of the ResourceType. In case of null, ResourceType will not be set.  
	 * @return The file created.
	 * 
	 * @throws CoreException
	 */
	public static IFile createResourceGroup(IFolder folder, String groupName, String systemId) throws CoreException {
		folder = folder.getFolder(groupName);
		if (!folder.exists()) {
			folder.create(true, true, null);
		}
		IFile processDefinitionFile = folder.getFile(ResourceGroup.RESOURCE_GROUP_FILENAME);
		ByteArrayInputStream groupBytes = createInitialResourceGroup(groupName, systemId);
		if (!processDefinitionFile.exists()) {
			processDefinitionFile.create(groupBytes, true, null);
		} else {
			processDefinitionFile.setContents(groupBytes, true, false, null);
		}
		return processDefinitionFile;
	}
	
	private static ByteArrayInputStream createInitialResourceGroup(String groupName, String systemId) throws JavaModelException {
		byte[] resourceGroupBytes = new byte[0];
		InputStream in = NewResourceGroupWizard.class.getResourceAsStream(ResourceGroup.RESOURCE_GROUP_FILENAME);
		try {
			String resourceGroup = new String(FileUtil.readFile(in));
			// Replaces the ResourceGroup name included in the "sample" file. 
			resourceGroup = resourceGroup.replaceAll(RESOURCE_GROUP_NAME_STRING, groupName);
			// Replaces the systemID included in the "sample" file to empty String
			// in case group is being created inside a Template project.
			if (systemId == null) {
				systemId = String.valueOf(ResourceType.DEFAULT_ID);
			}
			resourceGroup = resourceGroup.replaceAll(SYSTEM_ID_STRING, systemId);
			resourceGroupBytes = resourceGroup.getBytes();
		} catch (IOException e) {
			logger.error("Error loading bundled ResourceGroup file", e);
		}
		
		return new ByteArrayInputStream(resourceGroupBytes);
	}
	
}
