/*
 * Created on Jul 17, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.template.graph;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import org.jbpm.gd.jpdl.model.EndState;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.StartState;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.graph.WatchPointGraphUtils;
import com.tandbergtv.watchpoint.studio.validation.graph.WorkflowTemplateGraph;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

import edu.uci.ics.jung.graph.Vertex;

/**
 * Rule that validates that the Workflow Template Graph contains all the required nodes (with the
 * correct multiplicity). Checks that the graph contains only one Start node, atleast one End node, and at
 * least one other type of node.
 * 
 * @author Vijay Silva
 */
public class RequiredTemplateNodesRule extends ValidationRule<WorkflowTemplateGraph>
{
	/**
	 * Validates that the Graph element contains the required nodes.
	 * 
	 * @param graph
	 *            The object being validated
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	@SuppressWarnings("unchecked")
	public List<ValidationMessage> validateRule(WorkflowTemplateGraph graph)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		int startNodeCount = 0;
		int endNodeCount = 0;
		boolean otherNodesExist = false;

		Set<Vertex> vertices = graph.getVertices();
		for (Vertex vertex : vertices)
		{
			NodeElement node = WatchPointGraphUtils.getWTVertexElement(vertex);

			if (node instanceof StartState)
			{
				startNodeCount++;
			}
			else if (node instanceof EndState)
			{
				endNodeCount++;
			}
			else
			{
				otherNodesExist = true;
			}
		}

		Object graphElement = WatchPointGraphUtils.getElement(graph);

		this.validateStartNodeCount(startNodeCount, messages, graphElement);
		this.validateEndNodeCount(endNodeCount, messages, graphElement);
		this.validateNodeCount(otherNodesExist, messages, graphElement);

		return messages;
	}

	/*
	 * Validate the Start Node Count (must equal 1).
	 */
	private void validateStartNodeCount(int count, List<ValidationMessage> messages, Object element)
	{
		/* Check if the Start node is missing or there are extra start nodes */
		if (count == 0)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.TEMPLATE_MISSING_START_NODE);
		}
		else if (count > 1)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.TEMPLATE_EXTRA_START_NODE);
		}
	}

	/*
	 * Validate the End Node Count (must be >= 1).
	 */
	private void validateEndNodeCount(int count, List<ValidationMessage> messages, Object element)
	{
		/* Check if the End node is missing */
		if (count == 0)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.TEMPLATE_MISSING_END_NODE);
		}
	}

	/*
	 * Validate the Node Count of all nodes other than the Start and End nodes (must be > 0).
	 */
	private void validateNodeCount(boolean nodeExists, List<ValidationMessage> messages,
			Object element)
	{
		/* Check if there are no other nodes present */
		if (!nodeExists)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, element, ValidationMessageCode.TEMPLATE_MISSING_NODE);
		}
	}
}
