package com.tandbergtv.watchpoint.studio.debugger.runtime.nodesimulation;

import java.util.Collection;
import java.util.List;

import org.jbpm.context.def.VariableAccess;
import org.jbpm.graph.def.Node;
import org.jbpm.graph.exe.ExecutionContext;

import com.tandbergtv.watchpoint.studio.debugger.model.NodeSimulationConfig;
import com.tandbergtv.watchpoint.studio.debugger.runtime.exception.NodeSimulationException;

public abstract class NodeSimulator {
	
	protected Node node;
	protected NodeSimulationConfig simulationConfig;
	protected ExecutionContext executionContext;

	protected NodeSimulationException exception;
	
	/**
	 * 		Starts the simulation.
	 * 
	 * @param context
	 */
	public final void startSimulation(ExecutionContext executionContext) {
		this.executionContext = executionContext;
		
		try {
			simulate();
		} catch (Exception e) {
			if (e instanceof NodeSimulationException) {
				this.exception = (NodeSimulationException) e;
			} else {
				e.printStackTrace();
				String message = "Unexpected error while trying to simulate the node.";
				this.exception = new NodeSimulationException(message, node, this, e);
			}
		}
	}
	/**
	 * 		Simulates the node execution.
	 * @throws Exception in case an exception occurs which causes the nodesimulation to fail.
	 */
	protected abstract void simulate() throws Exception;
	
	/**
	 * 		Returns the transitions that should be taken after applying the simulation.
	 * Implementors may override if the node simulation changes the simulation path.
	 * 
	 * @return
	 */
	public List<?> getLeavingTransitions() {
		List<?> leavingTransitions = null;
		if (node != null) {
			leavingTransitions = node.getLeavingTransitions();
		}
		return leavingTransitions;
	}
	
	/**
	 * 		Tells if the node simulation has failed - which can happen because
	 * the simulation was configured to fail - or any exception occurred during 
	 * the simulation.
	 * 
	 * @return
	 */
	public boolean fail() {
		return exception != null;
	}
	
	/**
	 * 		Returns any exception that may have occurred during the node simulation.
	 * 
	 * @return
	 */
	public NodeSimulationException getException() {
		return exception;
	}

	public Node getNode() {
		return node;
	}

	public void setNode(Node node) {
		this.node = node;
	}

	public NodeSimulationConfig getSimulationConfig() {
		return simulationConfig;
	}

	public void setSimulationConfig(NodeSimulationConfig simulationConfig) {
		this.simulationConfig = simulationConfig;
	}
	
	/**
	 * 		Variables that affect the simulation
	 * @return
	 */
	public abstract Collection<VariableAccess> getVariables();

}
