package com.tandbergtv.watchpoint.studio.debugger.ui;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.CheckboxCellEditor;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableEditor;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.jbpm.gd.jpdl.model.Decision;

import com.tandbergtv.watchpoint.studio.debugger.model.NodeSimulationConfig;
import com.tandbergtv.watchpoint.studio.debugger.model.SimulationType;
import com.tandbergtv.watchpoint.studio.ui.model.AutomaticTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;

/**
 * 		The Table that shows the Template nodes.
 *  
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio - Francisco Bento da Silva Neto</a>
 *
 */
public class NodeSimulationTable {
	
	private Table table;
	private TableViewer tableViewer;
	protected String lastSortedColumn;
	protected boolean sortDesc;
	
	private List<SimulationModifyListener> simulationModifyListeners = new ArrayList<SimulationModifyListener>();

	private List<Button> skipButtons = new ArrayList<Button>();
	
	public final static String NODE_TYPE_COLUMN = "node_type";
	public final static String NODE_NAME_COLUMN = "name";
	public final static String NODE_DATA_COLUMN = "data";
	public final static String SKIP_COLUMN = "skip";
	public final static String SIMULATION_TYPE_COLUMN = "simulation_type";
	public static final String[] COLUMN_NAMES = new String[] { 
			NODE_TYPE_COLUMN,
			NODE_NAME_COLUMN,
			NODE_DATA_COLUMN,
			SIMULATION_TYPE_COLUMN,
			SKIP_COLUMN
	};
	
	public NodeSimulationTable(Composite parent) {
		this.createControls(parent);
	}
	
	/**
	 * 		Adds the Table controls to the composite.
	 * 
	 * @param composite
	 */
	private void createControls(Composite composite) {
		// Create a composite to hold the children
		GridData gridData = new GridData (GridData.HORIZONTAL_ALIGN_FILL | GridData.FILL_BOTH);
		composite.setLayoutData (gridData);

		// Set numColumns to 3 for the buttons 
		GridLayout layout = new GridLayout(1, false);
		layout.marginWidth = 4;
		composite.setLayout(layout);

		// Create the table 
		createTable(composite);
		// Create and setup the TableViewer
		createTableViewer();
	}
	
	public void setSimulationData(Collection<NodeSimulationConfig> nodeSimulationData) {
		tableViewer.setInput(nodeSimulationData);
		createSkipColunnEditor();
	}
	
	@SuppressWarnings("unchecked")	
	public void createSkipColunnEditor() {
		Collection<NodeSimulationConfig> nodeSimulationData = (Collection<NodeSimulationConfig>) tableViewer.getInput();
		
		for (Button btn : skipButtons) {
			btn.dispose();
		}
		int i = 0;
		skipButtons = new ArrayList<Button>(nodeSimulationData.size());
		for (final NodeSimulationConfig node : nodeSimulationData) {
			TableEditor checkboxEditor = new TableEditor(table);
			Button result = new Button(table, SWT.CHECK);
			result.addSelectionListener(new SelectionListener() {
				public void widgetSelected(SelectionEvent e) {
					Button btn = (Button) e.getSource();
					NodeSimulationConfig node = (NodeSimulationConfig) btn.getData();
					if (btn.getSelection()) {
						node.setSimulationType(SimulationType.SKIP_NODE);
					} else { 
						// if its 'unskipping' the node, select the next default option for the node.
						if (node.getNodeElement() instanceof AutomaticTaskNode || node.getNodeElement() instanceof LoopNode) {
							node.setSimulationType(SimulationType.EXECUTE);
						} else {
							node.setSimulationType(SimulationType.SIMULATED_VALUES);
						}
					}
					tableViewer.update(node, null);
					fireSimulationModifyListener(node);
				}
				
				public void widgetDefaultSelected(SelectionEvent e) {
				}
			});
			result.pack();
			checkboxEditor.minimumWidth = result.getSize ().x;
			checkboxEditor.horizontalAlignment = SWT.CENTER;

			checkboxEditor.setEditor(result, table.getItem(i), 4);
			
			if (node.getNodeElement() instanceof Decision) {
				result.setVisible(false);
			}
			result.setData(node);
			result.setSelection(SimulationType.SKIP_NODE.equals(node.getSimulationType()));
			skipButtons.add(result);
			
			i++;
		}
		
	}
	public void refreshSkipColumn() {
		int i = 0;
		for (TableItem item : table.getItems()) {
			NodeSimulationConfig node = (NodeSimulationConfig) item.getData();
			
			Button btn = skipButtons.get(i);
			if (node.getNodeElement() instanceof Decision) {
				btn.setVisible(false);
			} else {
				btn.setVisible(true);
			}
			
			btn.setData(node);
			btn.setSelection(SimulationType.SKIP_NODE.equals(node.getSimulationType()));
			i++;
		}
	}
	
	/**
	 * Create the Table
	 */
	private void createTable(Composite parent) {
		int style = SWT.SINGLE | SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | 
					SWT.FULL_SELECTION | SWT.HIDE_SELECTION;

		table = new Table(parent, style);
		
		GridData gridData = new GridData(GridData.FILL_BOTH);
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);		
					
		table.setLinesVisible(true);
		table.setHeaderVisible(true);

		// 1st column with Node Type
		TableColumn column = new TableColumn(table, SWT.LEFT, 0);
		column.setText("Node Type");
		column.setWidth(150);
		// Add listener for sorting the column  
		column.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				sortTable(NODE_TYPE_COLUMN);
			}
		});

		// 2nd column with Node Name
		column = new TableColumn(table, SWT.LEFT, 1);
		column.setText("Node Name");
		column.setWidth(150);
		column.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				sortTable(NODE_NAME_COLUMN);
			}
		});
		
		// 3rd column with the Node Data
		column = new TableColumn(table, SWT.LEFT, 2);
		column.setText("Node Data");
		column.setWidth(300);
		column.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				sortTable(NODE_DATA_COLUMN);
			}
		});

		// 4th column with Simulation Type text 
		column = new TableColumn(table, SWT.LEFT, 3);
		column.setText("Simulation Type");
		column.setWidth(100);
		column.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				sortTable(SIMULATION_TYPE_COLUMN);
			}
		});
		
		// 5th column with Skip checkbox 
		column = new TableColumn(table, SWT.CENTER, 4);
		column.setText("Skip");
		column.setWidth(50);
	}

	protected void sortTable(String column) {
		if (column.equals(lastSortedColumn)) {
			sortDesc = !sortDesc;
		} else {
			sortDesc = false;
		}
		NodeSimulationTableComparator comparator = new NodeSimulationTableComparator(column, sortDesc);
		tableViewer.setComparator(comparator);
		lastSortedColumn = column;
		refreshSkipColumn();
	}
	
	/**
	 * Create the TableViewer 
	 */
	private void createTableViewer() {
		tableViewer = new TableViewer(table);
		tableViewer.setUseHashlookup(true);
		tableViewer.setColumnProperties(COLUMN_NAMES);

		// Create the cell editors
		CellEditor[] editors = new CellEditor[COLUMN_NAMES.length];
		// Column 4 : Skip
		editors[4] = new CheckboxCellEditor(table);

		tableViewer.setContentProvider(new ArrayContentProvider());
		tableViewer.setLabelProvider(new NodeSimulationTableLabelProvider());
	}

	/*
	 * Close the window and dispose of resources
	 */
	public void close() {
		Shell shell = table.getShell();

		if (shell != null && !shell.isDisposed())
			shell.dispose();
	}

	/**
	 * Return the column names in a collection
	 * 
	 * @return List  containing column names
	 */
	public List<String> getColumnNames() {
		return Arrays.asList(COLUMN_NAMES);
	}

	/**
	 * @return currently selected item
	 */
	public NodeSimulationConfig getSelection() {
		NodeSimulationConfig config = null;
		
		TableItem[] selectedItems = table.getSelection();
		if (selectedItems.length > 0) {
			config = (NodeSimulationConfig) selectedItems[0].getData();
		}
		return config;
	}

	/**
	 * Return the parent composite
	 */
	public Control getControl() {
		return table.getParent();
	}

	public void updateNode(NodeSimulationConfig node) {
		tableViewer.update(node, null);
		for (Button button : skipButtons) {
			NodeSimulationConfig buttonNode = (NodeSimulationConfig) button.getData();
			if (buttonNode.getTemplateNodeName().equals(node.getTemplateNodeName())) {
				button.setSelection(SimulationType.SKIP_NODE.equals(node.getSimulationType()));
			}
		}
	}
	
	public void fireSimulationModifyListener(Object o) {
		for (SimulationModifyListener listener : simulationModifyListeners) {
			listener.modifySimulation(o);
		}
	}
	
	public void addSelectionListener(SelectionListener listener) {
		table.addSelectionListener(listener);
	}
	
	public void removeSelectionListener(SelectionListener listener) {
		table.removeSelectionListener(listener);
	}
	
	/**
	 * Release resources
	 */
	public void dispose() {
		// Tell the label provider to release its ressources
		tableViewer.getLabelProvider().dispose();
	}
	
	public void addSimulationModifyListener(SimulationModifyListener listener) {
		simulationModifyListeners.add(listener);
	}
	
}
