/*
 * Created on Jun 25, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess;

import java.util.Collection;
import java.util.List;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;

/**
 * Data Access Interface for the WorkflowTemplateDTO Entity
 * 
 * @author Vijay Silva
 */
public interface WorkflowTemplateDTODAI extends DataAccessInterface<WorkflowTemplateDTO, Long>
{
	/**
	 * Get the list of active Templates that have the most recent version number.
	 * 
	 * @return The List of active Templates that have the more recent version number.
	 */
	List<WorkflowTemplateDTO> findByCurrentVersion();

	/**
	 * Get the list of active Templates that are using the messageUID whose is passed.
	 * 
	 * @param messageUID
	 *            the id of the node definition whose usage is being checked
	 * @param type
	 * 			  the message type: SuperState or SingleNode
	 * @return The List of active Templates that are using the node definition
	 */
	List<WorkflowTemplateDTO> findByMessageUsage(Collection<String> messageUIDs, NodeDefinitionType type);

	/**
	 * 	Get the list of active Templates that are using the SuperState node whose node definition name and systemID is passed.
	 * 
	 * @param nodeDefinitionName
	 * @param systemID
	 * @return
	 */
	List<WorkflowTemplateDTO> findBySuperStateUsage(String nodeDefinitionName, String systemID);
	
	/**
	 * Get the template by its name
	 * 
	 * @param name
	 *            The Template Name
	 * 
	 * @return The Templates with matching name.
	 */
	List<WorkflowTemplateDTO> findByName(String name);
	
	/**
	 * Get the templates for a specified project.
	 * 
	 * @param projectName
	 * @return
	 */
	List<WorkflowTemplateDTO> findByProjectName(String projectName);
	
	/**
	 * Get the Template by its Path.
	 * 
	 * @param name
	 *            The Template Path
	 * 
	 * @return Template with matching path.
	 */
	WorkflowTemplateDTO findByPath(String path);

	/**
	 * Get the Id of the most recent version of an active Template by name.
	 * 
	 * @param name
	 *            The template name to match against.
	 * 
	 * @return The Id of the matching template.
	 */
	Long getTemplateIdByName(String name);

	/**
	 * Get the Id of the oldest version of an active Template by name.
	 * 
	 * @param name
	 *            The template name to match against.
	 * 
	 * @return The Id of the matching template.
	 */
	Long getTemplateIdByNameForOldestVersion(String name);

	/**
	 * Get the most recent version number of the an active Template with a matching name.
	 * 
	 * @param name
	 *            The Template name
	 * @return The current version number of the template (or null if template doesn't exist)
	 */
	Integer getCurrentVersionNumberByName(String name);

	/**
	 * Get the count of the number of active templates with matching name
	 * 
	 * @param name
	 *            The template name to match against.
	 * 
	 * @return The Count of matching active templates.
	 */
	int getCountByName(String name);

	/**
	 * Get the count of active Templates that are using the resource group specified.
	 * 
	 * @param resourceGroupName
	 *            the ResourceGroup name
	 * 
	 * @return The count of active Templates that are using the Resource Group.
	 */
	int getCountByResourceGroupUsage(String resourceGroupName);

	/**
	 * Deletes all references to this node definition by templates.
	 * 
	 * @param nodeDefinitionId
	 *            the key of the node definition hwose references are to be deleted
	 */
	void deleteNodeDefinitionReferences(long nodeDefinitionId);

	/**
	 * Get all Templates that refers to a specified template.
	 * @param subProcessName
	 * @return Templates that uses the specified template.
	 */
	List<WorkflowTemplateDTO> findByTemplateUsage(String subProcessName);
	
	/**
	 * Get all Templates that refers to a specified ResourceType.
	 * @param systemID
	 * @return Templates that uses the specified ResourceType.
	 */
	List<WorkflowTemplateDTO> findByResourceTypeUsage(String systemID);
	
	/**
	 * Get all Templates that refers to a specified ResourceGroup.
	 * @param groupName
	 * @return Templates that uses the specified ResourceGroup.
	 */
	List<WorkflowTemplateDTO> findByResourceGroupUsage(String groupName);
}
