/**
 * ExpressionsFixer.java
 * Created Feb 28, 2012
 */
package com.tandbergtv.watchpoint.studio.external.wpexport.impl.template;

import static com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType.SuperState;
import static com.tandbergtv.watchpoint.studio.util.ConvertExprUtility.convertDecisionExpression;
import static com.tandbergtv.watchpoint.studio.util.ConvertExprUtility.convertTimerExpression;

import org.jbpm.gd.jpdl.model.Condition;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.ui.model.AutomaticTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.IDueDateElement;
import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;

/**
 * Converts expressions in decisions and in timeout settings to the appropriate syntax
 */
public class ExpressionsFixer {
	
	/**
	 * Fixes expressions in all nodes in the container
	 * @param container
	 */
	public static void fixExpressions(NodeElementContainer container) {
		NodeElement [] childNodes = container.getNodeElements();
		
		if (childNodes == null) // Check if no child nodes are present
			return;

		for (NodeElement node : childNodes)
			fixNodeExpressions(node, container);
	}

	/**
	 * Goes through a node and converts all expressions
	 */
	private static void fixNodeExpressions(NodeElement nodeElement, NodeElementContainer container) {
		fixTimeoutExpression(nodeElement);

		if (nodeElement instanceof Decision)
			fixDecisionExpression((Decision) nodeElement);
		else if (nodeElement instanceof LoopNode)
			fixLoopNodeExpression((LoopNode) nodeElement);
		else if (nodeElement instanceof NodeDefinition) {
			NodeDefinition nodeDefinition = (NodeDefinition) nodeElement;
			
			if (nodeDefinition.getNodeType() == SuperState)
				fixExpressions(nodeDefinition);
		}

		/* FIXME Isn't this just for decisions? */
		fixTransitionConditionExpression(nodeElement);
	}
	
	private static void fixTimeoutExpression(NodeElement nodeElement) {
		// Automatic task node by itself does not have any due date and time duration
		if (!(nodeElement instanceof AutomaticTaskNode)) {
			if (nodeElement instanceof IDueDateElement) {
				IDueDateElement dueDate = (IDueDateElement) nodeElement;
				String oldDueDate = dueDate.getDueDate();

				if (oldDueDate != null && oldDueDate.trim().length() > 0)
					dueDate.setDueDate(convertTimerExpression(oldDueDate.trim()));
			}
		}
	}
	
	private static void fixDecisionExpression(Decision decision) {
		String expression = decision.getExpression();
		
		if (expression != null && expression.trim().length() > 0)
			decision.setExpression(convertDecisionExpression(expression));
	}
	
	private static void fixLoopNodeExpression(LoopNode loopNode) {
		String expression = loopNode.getExpression();
		
		if (expression != null && expression.trim().length() > 0)
			loopNode.setExpression(convertDecisionExpression(expression));
		
		fixExpressions(loopNode);
	}
	
	private static void fixTransitionConditionExpression(NodeElement nodeElement) {
		for (Transition transition : nodeElement.getTransitions()) {
			Condition condition = transition.getCondition();
			String expression = (condition != null) ? condition.getExpression() : null;
			
			if (expression != null && expression.trim().length() > 0)
				condition.setExpression(convertDecisionExpression(expression));
		}
	}
}
