package com.tandbergtv.watchpoint.studio.ui.model;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jbpm.gd.common.model.GenericElement;
import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.model.ActionElement;
import org.jbpm.gd.jpdl.model.Controller;
import org.jbpm.gd.jpdl.model.Event;
import org.jbpm.gd.jpdl.model.Task;

import com.tandbergtv.watchpoint.studio.application.ApplicationProperties;
import com.tandbergtv.watchpoint.studio.application.ApplicationPropertyKeys;
import com.tandbergtv.watchpoint.studio.util.SemanticElementCloneUtil;

/**
 * Model representation of an Automatic TaskNode.
 * 
 * @author Imran Naqvi
 * 
 */
public class AutomaticTaskNode extends AbstractTaskNode implements IMessageContainer, IWPVariableContainer {

	private static final String VALUE_ELEMENT_NAME = "value";

	private static final String KEY_ELEMENT_NAME = "key";

	private static final String KEYS_ELEMENT_NAME = "keys";

	private static final String ENTRY_ELEMENT_NAME = "entry";

	private static final String MAP_ELEMENT_NAME = "map";
	
	private static final String ROLLBACK_TASK_NAME = "undo";
	
	private static final String UNDO_EVENT_TYPE = "node-undo";

	private Map<String, String> keys = null;

	// ========================================================================
	// ===================== IMessageContainer METHODS
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getUid()
	 */
	public String getUid() {
		Action action = this.getAction();
		if (action == null)
			return null;
		GenericElement sendElement = null;
		for (GenericElement element : action.getGenericElements()) {
			/* Check for either send or receive element, there should be only one */
			if (element.getName().equals("send") || element.getName().equals("receive")) {
				sendElement = element;
				break;
			}
		}
		if (sendElement == null)
			return "";

		return sendElement.getValue();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#setUid(java.lang.String)
	 */
	public void setUid(String uid) {
		if (this.getAction() == null)
			setAction(createAction());

		String elementName = this.isOutgoingMessage(uid) ? "send" : "receive";
		String elementToRemove = this.isOutgoingMessage(uid) ? "receive" : "send";

		Action action = this.getAction();
		GenericElement sendElement = null;
		for (GenericElement element : action.getGenericElements()) {
			if (element.getName().equals(elementName)) {
				sendElement = element;
			}
			else if (element.getName().equals(elementToRemove)) {
				action.removeGenericElement(element);
			}
		}
		if (sendElement == null) {
			sendElement = createWPConfigInfoElement();
			sendElement.setName(elementName);
			action.addGenericElement(sendElement);
		}

		sendElement.setValue(uid);
		this.getAction().setClassName(getHandler());
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#setRollbackNode(com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition)
	 */
	public void setRollbackNode(NodeDefinition rollbackNode) {
		String elementName = this.isOutgoingMessage(rollbackNode.getUid()) ? "send" : "receive";
		String elementToRemove = this.isOutgoingMessage(rollbackNode.getUid()) ? "receive" : "send";
		
		Event undoEvent = getEvent(UNDO_EVENT_TYPE);
		if (undoEvent != null) {
			removeEvent(undoEvent);
		}
		undoEvent = createEvent(UNDO_EVENT_TYPE);
		
		Action action = createAction();
		GenericElement sendElement = null;
		for (GenericElement element : action.getGenericElements()) {
			if (element.getName().equals(elementName)) {
				sendElement = element;
			}
			else if (element.getName().equals(elementToRemove)) {
				action.removeGenericElement(element);
			}
		}
		if (sendElement == null) {
			sendElement = createWPConfigInfoElement();
			sendElement.setName(elementName);
			action.addGenericElement(sendElement);
		}

		sendElement.setValue(rollbackNode.getUid());
		
		action.setName(rollbackNode.getName());
		action.setClassName(elementName);
		
		undoEvent.addActionElement(action);
		
		addEvent(undoEvent);
		
		setRollbackVariables(rollbackNode.getVariables());
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getRollbackUid()
	 */
	public String getRollbackUid() {
		String rollbackUID = null;
		Event undoEvent = getEvent(UNDO_EVENT_TYPE);
		if (undoEvent != null) {
			ActionElement[] actions = undoEvent.getActionElements();
			for (ActionElement action : actions) {
				if (action instanceof Action) {
					for (GenericElement element : ((Action)action).getGenericElements()) {
						rollbackUID = element.getValue();
						break;
					}	
				}
			}
		}
		return rollbackUID;
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getRollbackDefinitionName()
	 */
	public String getRollbackDefinitionName() {
		String definitionName = null;
		Event undoEvent = getEvent(UNDO_EVENT_TYPE);
		if (undoEvent != null) {
			ActionElement[] actions = undoEvent.getActionElements();
			for (ActionElement action : actions) {
				if (action instanceof Action) {
					definitionName = action.getName();
						break;
				}
			}
		}
		return definitionName;
	}
	
	public void removeRollbackNode() {
		Event undoEvent = getEvent(UNDO_EVENT_TYPE);
		if (undoEvent != null) {
			removeEvent(undoEvent);
		}
		Task task = getRollbackTask();
		if (task != null) {
			removeTask(task);
		} 
	}
	
	private Event createEvent(String eventType) {
		Event event = (Event) getFactory().createById(SemanticElementConstants.EVENT_SEID);
		event.setType(eventType);
		return event;
	}
	
	private Event getEvent(String eventType) {
		Event undoEvent = null;
		Event[] events = super.getEvents();
		for(Event event : events) {
			if (eventType.equals(event.getType())) {
				undoEvent = event;
				break;
			}
		}
		return undoEvent;
	}
	

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getIncomingVariables()
	 */
	public List<WPVariable> getIncomingVariables() {
		List<WPVariable> incomingVars = new ArrayList<WPVariable>();
		/* return empty list if this node deosn't have any tasks or the default task couldn't be found */
		Task task = getDefaultTask();
		if (task == null)
			return incomingVars;
		for (int i = 0; i < task.getController().getVariables().length; i++) {
			WPVariable var = (WPVariable) task.getController().getVariables()[i];
			if (var.isWritable())
				incomingVars.add(var);
		}
		return incomingVars;
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getIncomingVariables()
	 */
	public List<WPVariable> getRollbackIncomingVariables() {
		List<WPVariable> incomingVars = new ArrayList<WPVariable>();
		/* return empty list if this node deosn't have any tasks or the default task couldn't be found */
		Task task = getRollbackTask();
		if (task == null)
			return incomingVars;
		for (int i = 0; i < task.getController().getVariables().length; i++) {
			WPVariable var = (WPVariable) task.getController().getVariables()[i];
			if (var.isWritable())
				incomingVars.add(var);
		}
		return incomingVars;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getOutgoingVariables()
	 */
	public List<WPVariable> getRollbackOutgoingVariables() {
		List<WPVariable> outgoingVars = new ArrayList<WPVariable>();
		/* return empty list if this node deosn't have any tasks or the default task couldn't be found */
		Task task = getRollbackTask();
		if (task == null)
			return outgoingVars;
		for (int i = 0; i < task.getController().getVariables().length; i++) {
			WPVariable var = (WPVariable) task.getController().getVariables()[i];
			if (var.isReadable())
				outgoingVars.add(var);
		}
		return outgoingVars;
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getOutgoingVariables()
	 */
	public List<WPVariable> getOutgoingVariables() {
		List<WPVariable> outgoingVars = new ArrayList<WPVariable>();
		/* return empty list if this node deosn't have any tasks or the default task couldn't be found */
		Task task = getDefaultTask();
		if (task == null)
			return outgoingVars;
		for (int i = 0; i < task.getController().getVariables().length; i++) {
			WPVariable var = (WPVariable) task.getController().getVariables()[i];
			if (var.isReadable())
				outgoingVars.add(var);
		}
		return outgoingVars;
	}
	
	/**
	 * 		Gets the default Task - the task that maps the NodeDefinition variables.
	 * 		For backward compatibility reasons, this task should have no name specified. 
	 * 
	 * @return
	 */
	private Task getDefaultTask() {
		Task[] tasks = getTasks();
		for ( Task task : tasks) {
			if ( task.getName() == null || task.getName().trim().isEmpty() ) {
				return task;
			}
		}
		return null;
	}
	
	/**
	 * 		Gets the undo Task - the task that maps the Rollback NodeDefinition variables.
	 * 		This task should have the "undo" name specified. 
	 * 
	 * @return
	 */
	private Task getRollbackTask() {
		Task[] tasks = getTasks();
		for ( Task task : tasks) {
			if ( ROLLBACK_TASK_NAME.equals(task.getName()) ) {
				return task;
			}
		}
		return null;
	}

	public List<WPVariable> getRollbackVariables() {
		List<WPVariable> variables = new ArrayList<WPVariable>();
		
		variables.addAll(getRollbackOutgoingVariables());
		variables.addAll(getRollbackIncomingVariables());
		
		return variables;
	}
	
    /* (non-Javadoc)
     * @see com.tandbergtv.watchpoint.studio.ui.model.IWPVariableContainer#getVariables()
     */
	public List<WPVariable> getVariables() {
		List<WPVariable> variables = new ArrayList<WPVariable>();
		
		variables.addAll(getOutgoingVariables());
		variables.addAll(getIncomingVariables());
		
		return variables;
	}

    /* (non-Javadoc)
     * @see com.tandbergtv.watchpoint.studio.ui.model.IWPVariableContainer#hasVariables()
     */
	public boolean hasVariable(String name) {
		for (WPVariable variable : getVariables()) {
			if (variable.getMappedName().equals(name))
				return true;
		}
		
		return false;
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#addVariable(com.tandbergtv.watchpoint.studio.ui.model.WPVariable)
	 */
	public void addVariable(WPVariable variable) {
		Task task = getDefaultTask();
		if (task == null)
			addTask(this.createTask());
		
		if (hasVariable(variable.getMappedName()))
			return;
		
		task = getDefaultTask();
		task.getController().addVariable(variable);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#addVariable(com.tandbergtv.watchpoint.studio.ui.model.WPVariable)
	 */
	private void setRollbackVariables(Collection<WPVariable> variables) {
		Task task = getRollbackTask();
		if (task != null) {
			removeTask(task);
		} 
		addTask(this.createRollbackTask());
		task = getRollbackTask();
		
		for(WPVariable var : variables) {
			task.getController().addVariable(var);
		}
	}

    /* (non-Javadoc)
     * @see com.tandbergtv.watchpoint.studio.ui.model.IWPVariableContainer#removeVariable(java.lang.String)
     */
	public WPVariable removeVariable(String varName) {
		/* return null if this node deosn't have any tasks */
		Task task = getDefaultTask();
		if (task == null)
			return null;
		for (int i = 0; i < task.getController().getVariables().length; i++) {
			WPVariable var = (WPVariable) task.getController().getVariables()[i];
			if (var.getMappedName().equals(varName)) {
				task.getController().removeVariable(var);
				/* Remove task if it has no variables */
				if (task.getController().getVariables().length == 0)
					removeTask(task);
				return var;
			}
		}

		return null;
	}
	
	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#setConstantValue(java.lang.String,
	 *      java.lang.String)
	 */
	public void setConstantValue(String varName, String val) {
		if (this.getAction() == null)
			setAction(createAction());

		setVariableValue(varName, val, MAP_ELEMENT_NAME);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getConstantValue(java.lang.String)
	 */
	public String getConstantValue(String varName) {
		if (this.getAction() == null)
			return null;

		return getVariableValue(varName, MAP_ELEMENT_NAME);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getConstants()
	 */
	public Map<String, String> getConstants() {
		return getVariables(MAP_ELEMENT_NAME);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getCompositeKey(java.lang.String)
	 */
	public String getCompositeKey(String varName) {
		if (this.getAction() == null)
			return "";

		return getVariableValue(varName, KEYS_ELEMENT_NAME);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#getCompositeKeys()
	 */
	public Map<String, String> getCompositeKeys() {
		if (keys != null)
			return keys;

		return getVariables(KEYS_ELEMENT_NAME);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#setCompositeKey(java.lang.String,
	 *      java.lang.String)
	 */
	public void setCompositeKey(String varName, String val) {
		if (this.getAction() == null)
			setAction(createAction());

		setVariableValue(varName, val, KEYS_ELEMENT_NAME);
	}

	/**
	 * {@inheritDoc}
	 */
	public void removeCompositeKey(String name) {
		Action action = getAction();
		
		if (action == null)
			return;
		
		for (GenericElement element : action.getGenericElements()) {
			if (KEYS_ELEMENT_NAME.equals(element.getName())) {
				for (GenericElement entryElement : element.getGenericElements()) {
					String keyname = getElementProperty(entryElement, VALUE_ELEMENT_NAME);
					if (keyname.equals(name)) {
						element.removeGenericElement(entryElement);
						break;
					}
				}
			}
		}
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#hasAttachment()
	 */
	public boolean hasAttachment() {
		for (WPVariable var : getOutgoingVariables()) {
			if (var.isAttachment())
				return true;
		}
		return false;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IMessageContainer#updateActionHandler()
	 */
	public void updateActionHandler() {
		if (this.getAction() == null)
			this.setAction(this.createAction());
		this.getAction().setClassName(getHandler());
	}

	// ========================================================================
	// ===================== END IMessageContainer METHODS
	// ========================================================================

	/*
	 * Get the list of variables given the name of the element
	 */
	private Map<String, String> getVariables(String elementName) {
		Action action = this.getAction();
		Map<String, String> variables = new HashMap<String, String>();
		GenericElement entryContainerElement = null;
		/* return empty list if this node deosn't have any tasks */
		if (action == null || action.getGenericElements() == null)
			return variables;
		for (GenericElement element : action.getGenericElements()) {
			if (element.getName().equals(elementName)) {
				entryContainerElement = element;
				break;
			}
		}
		if (entryContainerElement == null)
			return variables;

		if (MAP_ELEMENT_NAME.equals(elementName)) {
			for (GenericElement entryElement : entryContainerElement.getGenericElements()) {
				variables.put(getElementProperty(entryElement, KEY_ELEMENT_NAME),
						getElementProperty(entryElement, VALUE_ELEMENT_NAME));
			}
		}
		/*
		 * For a list of keys we need to use the value as the key in out map since the configElement
		 * key would be empty.
		 */
		else if (KEYS_ELEMENT_NAME.equals(elementName)){
			for (GenericElement entryElement : entryContainerElement.getGenericElements()) {
				variables.put(getElementProperty(entryElement, VALUE_ELEMENT_NAME),
						getElementProperty(entryElement, KEY_ELEMENT_NAME));
			}
		}
		return variables;
	}

	/*
	 * Sets the constant or the composite key depending on the element name that is passed.
	 * 
	 * @param varName @param val @param elementName
	 */
	private void setVariableValue(String varName, String val, String elementName) {
		Action action = this.getAction();
		GenericElement entryContainerElement = null;
		if (action != null && action.getGenericElements() != null) {
			for (GenericElement element : action.getGenericElements()) {
				if (element.getName().equals(elementName)) {
					entryContainerElement = element;
					break;
				}
			}
		}

		if (entryContainerElement == null) {
			entryContainerElement = createWPConfigInfoElement();
			entryContainerElement.setName(elementName);
			action.addGenericElement(entryContainerElement);
		}

		GenericElement entryElement = null;
		for (GenericElement element : entryContainerElement.getGenericElements()) {
			if (element.getName().equals(ENTRY_ELEMENT_NAME)) {
				if ((MAP_ELEMENT_NAME.equals(elementName) && varName.equals(getElementProperty(
						element, KEY_ELEMENT_NAME)))
						|| (KEYS_ELEMENT_NAME.equals(elementName) && varName.equals(getElementProperty(
								element, VALUE_ELEMENT_NAME)))) {
					entryElement = element;
					break;
				}
			}
		}
		/* If value is null or blank we remove the configInfoElements and return */
		if (val == null) {
			if (entryElement != null)
				entryContainerElement.removeGenericElement(entryElement);
			if (entryContainerElement.getGenericElements() == null
					|| entryContainerElement.getGenericElements().length == 0)
				action.removeGenericElement(entryContainerElement);
			return;
		}
		/* Otherwise we set the value of the configInfoElement */
		if (entryElement == null) {
			entryElement = createWPConfigInfoElement();
			entryElement.setName(ENTRY_ELEMENT_NAME);
			entryContainerElement.addGenericElement(entryElement);
		}
		if (MAP_ELEMENT_NAME.equals(elementName))
			setEntryElementValue(entryElement, varName, val);
		else if (KEYS_ELEMENT_NAME.equals(elementName))
			setEntryElementValue(entryElement, val, varName);
	}

	/*
	 * Gets the constant or the composite key value depending on the element name that is passed.
	 * 
	 * @param varName @param elementName @return
	 */
	private String getVariableValue(String varName, String elementName) {
		Action action = this.getAction();

		GenericElement entryContainerElement = null;
		if (action != null && action.getGenericElements() != null) {
			for (GenericElement element : action.getGenericElements()) {
				if (element.getName().equals(elementName)) {
					entryContainerElement = element;
					break;
				}
			}
		}

		if (entryContainerElement == null)
			return null;
		for (GenericElement entryElement : entryContainerElement.getGenericElements()) {
			if (entryElement.getName().equals(ENTRY_ELEMENT_NAME))
				if (MAP_ELEMENT_NAME.equals(elementName)
						&& varName.equals(getElementProperty(entryElement, KEY_ELEMENT_NAME))) {
					for (GenericElement element : entryElement.getGenericElements()) {
						if (element.getName().equals(VALUE_ELEMENT_NAME))
							return element.getValue();
					}
					break;
				}
				else if (KEYS_ELEMENT_NAME.equals(elementName)
						&& varName.equals(getElementProperty(entryElement, VALUE_ELEMENT_NAME))) {
					for (GenericElement element : entryElement.getGenericElements()) {
						if (element.getName().equals(KEY_ELEMENT_NAME))
							return element.getValue();
					}
					break;
				}
		}
		return null;
	}

	/*
	 * Sets the entry element value.
	 * 
	 * @param entryElement @param varName @param val
	 */
	private void setEntryElementValue(GenericElement entryElement, String varName, String val) {
		GenericElement keyElement = null;
		GenericElement valueElement = null;
		for (GenericElement element : entryElement.getGenericElements()) {
			if (element.getName().equals(KEY_ELEMENT_NAME))
				keyElement = element;
			else if (element.getName().equals(VALUE_ELEMENT_NAME))
				valueElement = element;
			if (keyElement != null && valueElement != null)
				break;
		}
		/* Add key element if its not there */
		if (keyElement == null) {
			keyElement = createWPConfigInfoElement();
			keyElement.setName(KEY_ELEMENT_NAME);
			entryElement.addGenericElement(keyElement);
		}

		if (keyElement.getValue() == null || !keyElement.getValue().equals(varName))
			keyElement.setValue(varName);

		/* Add value element if its not there */
		if (valueElement == null) {
			valueElement = createWPConfigInfoElement();
			valueElement.setName(VALUE_ELEMENT_NAME);
			entryElement.addGenericElement(valueElement);
		}

		valueElement.setValue(val);
	}

	/*
	 * Gets the value of the property parameter of the element.
	 * 
	 * @param element @return
	 */
	private String getElementProperty(GenericElement element, String property) {
		GenericElement keyElement = null;
		for (GenericElement subElement : element.getGenericElements()) {
			if (subElement.getName().equals(property)) {
				keyElement = subElement;
				break;
			}
		}
		if (keyElement == null)
			return null;

		return keyElement.getValue();
	}

	/*
	 * Creates a new instance of WPConfigInfoElement
	 * 
	 * @return a new instance of WPConfigInfoElement
	 */
	private GenericElement createWPConfigInfoElement() {
		return (GenericElement) this.getFactory().createById(
				SemanticElementConstants.GENERIC_ELEMENT_SEID);
	}

	/*
	 * Creates a new task with a controller.
	 * 
	 * @return a new task with a controller
	 */
	private Task createTask() {
		Controller controller = (Controller) this.getFactory().createById(
				SemanticElementConstants.CONTROLLER_SEID);
		Task task = (Task) this.getFactory().createById(SemanticElementConstants.TASK_SEID);
		task.setController(controller);
		return task;
	}
	
	/*
	 * Creates a new task with a controller.
	 * 
	 * @return a new task with a controller
	 */
	private Task createRollbackTask() {
		Controller controller = (Controller) this.getFactory().createById(
				SemanticElementConstants.CONTROLLER_SEID);
		Task task = (Task) this.getFactory().createById(SemanticElementConstants.TASK_SEID);
		task.setController(controller);
		task.setName(ROLLBACK_TASK_NAME);
		return task;
	}

	private Action createAction() {
		Action action = (Action) this.getFactory().createById(SemanticElementConstants.ACTION_SEID);
		return action;
	}

	private String getHandler() {
		String outgoingHandler = ApplicationProperties.getInstance().get(
				ApplicationPropertyKeys.AUTO_TASK_OUTGOINGHANDLER).toString();
		String incomingHandler = ApplicationProperties.getInstance().get(
				ApplicationPropertyKeys.AUTO_TASK_INCOMINGHANDLER).toString();
		String attachmentHandler = ApplicationProperties.getInstance().get(
				ApplicationPropertyKeys.AUTO_TASK_ATTACHMENTHANDLER).toString();
		if (this.hasAttachment())
			return attachmentHandler;
		return this.isOutgoingMessage(getUid()) ? outgoingHandler : incomingHandler;
	}

	private boolean isOutgoingMessage(String uid) {
		String workflowSystemId = ApplicationProperties.getInstance().get(
				ApplicationPropertyKeys.WORKFLOW_SYSTEM_ID).toString();
		return !uid.substring(2, 4).equals(workflowSystemId);
	}
	
	public AutomaticTaskNode cloneAutomaticTaskNode(){
		return SemanticElementCloneUtil.cloneAutomaticTaskNode(this);
	}
	
}
