package com.tandbergtv.watchpoint.studio.ui.wizard;

import static com.tandbergtv.watchpoint.studio.ui.preferences.PreferenceConstants.EXPORT_DIR;
import static java.io.File.separator;
import static org.eclipse.swt.SWT.SYSTEM_MODAL;

import java.io.File;

import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.FormAttachment;
import org.eclipse.swt.layout.FormData;
import org.eclipse.swt.layout.FormLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;

import com.tandbergtv.watchpoint.studio.ui.UI;
import com.tandbergtv.watchpoint.studio.ui.WatchPointStudioPlugin;

public class ExportSelectDirPage extends AbstractWizardPage {

	private static final String NOT_A_FOLDER = UI.ExportSelectDirPage_NotFolder;

	private static final String SELECT_FOLDER = UI.ExportSelectDirPage_SelectFolder;

	private static final String BROWSE = UI.ExportSelectDirPage_Browse;

	private static final String EMPTY_STRING = ""; //$NON-NLS-1$
	
	private static final String DIRECTORY = "directory"; //$NON-NLS-1$
		
	private Combo directoryCombo;
	
	private Label helpLabel;
	
	private String suffix;

	private Button publishButton;

	private Button browseButton;
	
	private Button directoryButton;
	
	private static final int MAXENTRIES = 5;
	
	public ExportSelectDirPage(String pageName) {
		super(pageName);
		setTitle(pageName);
	}

	/**
	 * @return the helpText
	 */
	public String getSuffix() {
		return suffix;
	}

	/**
	 * @param helpText the helpText to set
	 */
	public void setSuffix(String helpText) {
		this.suffix = helpText;
	}
	
	/**
	 * Calculate the path to export to. If the default publish location is chosen, 
	 * generate the fully-qualified path. Otherwise take whatever directory has been selected.
	 * 
	 * @return
	 */
	public String getSelectedDir() {
		String text = EMPTY_STRING;
		
		if (this.publishButton.getSelection())
			text = getPreferredExportLocation();
		else
			text = this.directoryCombo.getText();
		
		return text;
	}
	
	@Override
	public boolean isPageComplete() {
		return (getErrorMessage() == null && getSelectedDir().length() > 0);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent) {
		initializeDialogUnits(parent);
		Composite composite = createClientArea(parent);

		createDirSelectionControl(composite);

		setControl(composite);
		initialize(getDialogSettings());
		Dialog.applyDialogFont(composite);
	}
	
	/*
	 * Sets the layout of the client area.
	 * 
	 * @param parent
	 *            the parent composite of the client area
	 * @return the composite representing the client area
	 */
	protected Composite createClientArea(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		FormLayout layout = new FormLayout();
		layout.marginWidth = 10;
		layout.marginHeight = 0;
		composite.setLayout(layout);
		return composite;
	}
	
	/**
	 * Creates and adds widgets to the control
	 * @param parent
	 */
	protected void createDirSelectionControl(Composite parent) {
		createPublishRadioButton(parent);
		createDirectoryRadioButton(parent);
		createBrowseButton(parent);
		createDirectoryCombo(parent);
		createHelpLabel(parent);
	}
	
	private void createPublishRadioButton(Composite parent) {
		FormData formData = new FormData();
		
		publishButton = new Button(parent, SWT.RADIO);
		
		publishButton.setEnabled(true);
		publishButton.setSelection(true);
		publishButton.setText(UI.ExportSelectDirPage_PublishLocation);
		formData.top = new FormAttachment(0);
		publishButton.setLayoutData(formData);
	}
	
	private void createDirectoryRadioButton(Composite parent) {
		FormData formData = new FormData();
		
		directoryButton = new Button(parent, SWT.RADIO);
		directoryButton.setText(UI.ExportSelectDirPage_Directory);
		directoryButton.setSelection(false);
		formData.top = new FormAttachment(publishButton, 10);
		directoryButton.setLayoutData(formData);
	}
	
	private void createBrowseButton(Composite parent) {
		FormData formData = new FormData();
		
		browseButton = new Button(parent, SWT.PUSH);
		browseButton.setText(BROWSE);
		browseButton.setEnabled(false);
		formData.top = new FormAttachment(directoryButton, 10);
		formData.right = new FormAttachment(100, -10);
		browseButton.setLayoutData(formData);
	}
	
	private void createDirectoryCombo(Composite parent) {
		FormData formData = new FormData();

		this.directoryCombo = new Combo(parent, SWT.DROP_DOWN);
		formData.top = new FormAttachment(directoryButton, 10);
		formData.right = new FormAttachment(browseButton, -10);
		formData.left = new FormAttachment(0, 5);
		directoryCombo.setLayoutData(formData);
	}
	
	private void createHelpLabel(Composite parent) {
		FormData formData = new FormData();
		
		helpLabel = new Label(parent, SWT.NONE);
		formData.top = new FormAttachment(this.directoryCombo, 10);
		formData.right = new FormAttachment(100, -5);
		formData.width = 500;
		helpLabel.setLayoutData(formData);
	}
	
	/**
	 * Initializes the control
	 */
	protected void initialize(IDialogSettings settings) {
		initializeCombo(settings, DIRECTORY, this.directoryCombo);
		String location = getPreferredExportLocation();
		
		if (this.directoryCombo.getItemCount() == 0) {
			this.directoryCombo.add(location);
			this.directoryCombo.setText(location);
		} else if (!isValidLocation(this.directoryCombo.getText())) {
			this.directoryCombo.setText(location);
		}
		
		hookListeners();
		updateControls();
		checkControls();
	}
	
	/**
	 * Attach listeners to widgets on the page
	 */
	protected void hookListeners() {
		publishButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if (publishButton.getSelection()) {
					updateControls();
					validate();
				}
			}
		});
		
		directoryButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				if (directoryButton.getSelection()) {
					updateControls();
					validate();
				}
			}
		});
		
		browseButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				String path = chooseDirectory(directoryCombo.getText());
				
				if (path != null) {
					if (directoryCombo.indexOf(path) == -1)
						directoryCombo.add(path, 0);
					directoryCombo.setText(path);
				}
			}
		});
		
		this.directoryCombo.addModifyListener(new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				updateControls();
				validate();
			}
		});
	}
	
	/**
	 * Determine if there are any errors on the page
	 */
	protected void checkControls() {
		String path = getSelectedDir();
		
		if (path.trim().length() == 0) {
			setErrorMessage(UI.ExportSelectDirPage_Empty);
			return;
		}
		
		if (path.trim().length() < path.length()) {
			setErrorMessage(UI.ExportSelectDirPage_InvalidName);
			return;
		}

		File file = new File(path);
		
		if (file.isFile()) {
			setErrorMessage(NOT_A_FOLDER);
			return;
		}
		
		setErrorMessage(null);
	}
	
	/**
	 * Validate and update wizard buttons
	 */
	private void validate() {
		checkControls();

		getContainer().updateButtons();
	}

	private String chooseDirectory(String path) {
		DirectoryDialog dialog = new DirectoryDialog(Display.getCurrent().getActiveShell(), SYSTEM_MODAL);
		
		dialog.setFilterPath(path);
		dialog.setText(SELECT_FOLDER);
		
		return dialog.open();
	}
	
	/**
	 * Updates state of controls on the page based on selection
	 */
	protected void updateControls() {
		this.browseButton.setEnabled(this.directoryButton.getSelection());
		this.directoryCombo.setEnabled(this.directoryButton.getSelection());
		this.helpLabel.setText(UI.ExportSelectDirPage_ExportLabel + getSelectedDir());
	}
	
	private String getPreferredExportLocation() {
		IPreferenceStore prefs = WatchPointStudioPlugin.getDefault().getPreferenceStore();
		
		return prefs.getString(EXPORT_DIR) + separator + this.suffix;
	}
	
	/**
	 * Returns whether the location text is a valid path
	 * @param location location path string
	 * @return true if the location is valid, false otherwise
	 */
	protected boolean isValidLocation(String location) {
		File file = new File(location);

		if (file.exists() && !file.isDirectory())
			return false;

		return true;
	}

	/**
	 * {@inheritDoc}
	 */
	protected void saveSettings(IDialogSettings settings) {
		saveCombo(settings, DIRECTORY, this.directoryCombo);
	}
	
	/**
	 * Initializes the combo box with all saved entries and highlights the first available one
	 * 
	 * @param settings
	 * @param key
	 * @param combo
	 */
	protected void initializeCombo(IDialogSettings settings, String key, Combo combo) {
		for (int i = 0; i < MAXENTRIES + 1; i++) {
			String text = settings.get(key + String.valueOf(i));
			
			if (text != null && combo.indexOf(text) == -1)
				combo.add(text);
		}
		
		if (combo.getItemCount() > 0)
			combo.setText(combo.getItem(0));
	}

	/**
	 * Saves most recent entries to the settings plus current
	 * 
	 * @param settings
	 * @param key
	 * @param combo
	 */
	protected void saveCombo(IDialogSettings settings, String key, Combo combo) {
		if (combo.getText().trim().length() > 0) {
			settings.put(key + String.valueOf(0), combo.getText().trim());
			String [] items = combo.getItems();
			int count = Math.min(items.length, MAXENTRIES);
			
			for (int i = 0; i < count; i++)
				settings.put(key + String.valueOf(i + 1), items[i].trim());
		}
	}
}
