package com.tandbergtv.watchpoint.studio.ui.wizard.imports;

import static com.tandbergtv.watchpoint.studio.ui.util.Utility.ID_RESOURCE_GROUP;
import static com.tandbergtv.watchpoint.studio.ui.util.Utility.ID_RESOURCE_TYPE;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.apache.log4j.Logger;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.resources.WorkspaceJob;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.java.plugin.PluginManager.PluginLocation;
import org.java.plugin.boot.DefaultPluginsCollector;
import org.java.plugin.boot.PluginsCollector;
import org.java.plugin.util.ExtendedProperties;
import org.w3c.dom.Document;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer;
import com.tandbergtv.watchpoint.studio.ui.wizard.ResourceTypeJavaProjectCreator;
import com.tandbergtv.watchpoint.studio.util.FileUtil;
import com.tandbergtv.watchpoint.studio.util.XMLDocumentUtility;

/**
 * Performs the import operation as a job
 * 
 * @author Patrik Araujo
 */
public class ResourceTypeImportOperation extends WorkspaceJob {
	
	private static final String REPOSITORIES = "org.java.plugin.boot.pluginsRepositories";
	private static final String CLASSES_FOLDER_NAME = "classes";
	private static final String LIB_FOLDER_NAME = "lib";
	private static final String CLASSES_JAR_FILENAME = "device";
	private static final String CLASSES_JAR_EXTENSION = ".jar";

	Collection<? extends IWatchPointDTO> models;
	String importLocation;
	Map<String,String> locationInfo;
	boolean overwriteExistingProjects = true;
	
	private static final Logger logger = Logger.getLogger(ResourceTypeImportOperation.class);
	
	/**
	 * @param name
	 * @param location 
	 * @param importer
	 * @param items
	 */
	public ResourceTypeImportOperation(String name, Collection<? extends IWatchPointDTO> models, String location, boolean overwriteProjects) {
		super(name);
		this.models = models;
		this.importLocation = location;
		this.overwriteExistingProjects = overwriteProjects;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public IStatus runInWorkspace(IProgressMonitor monitor) throws CoreException {
		monitor.beginTask("Importing items", this.models.size() + 1);
		
		try {
			importResourceTypes(this.models, monitor);
			
			if (monitor.isCanceled())
				return Status.CANCEL_STATUS;
		} catch (Exception e) {
			logger.error("Import failed", e);
			IStatus status = new Status(IStatus.ERROR, Utility.PLUGIN_ID, "Import failed", e.getCause());
			throw new CoreException(status);
		} finally {
			monitor.done();
			updateDisplay();
		}
		
		return Status.OK_STATUS;
	}
	
	private void importResourceTypes(Collection<? extends IWatchPointDTO> models, IProgressMonitor monitor) 
					throws JavaModelException, CoreException, IOException, InvocationTargetException, InterruptedException {
		// Creates a map of resource types and their respective locations
		buildLocationInfo();
		
		// Create project for the resource type
		createResourceTypeProject(models, monitor);
	}
	
	private void createResourceTypeProject(Collection<? extends IWatchPointDTO> models, IProgressMonitor monitor) throws JavaModelException, CoreException, IOException, InvocationTargetException, InterruptedException{
		for (IWatchPointDTO resourceTypeDTO : models) {
			ResourceType resourceType = (ResourceType)resourceTypeDTO;
			monitor.setTaskName("Importing " + resourceType.getName() + " project.");
			
			if(overwriteExistingProjects){
				// Check if project exists and delete it if true
				checkIfProjectExistsAndDelete(resourceType);
			}
			
			// Creates the project for each resource type being imported
			ResourceTypeJavaProjectCreator creator = new ResourceTypeJavaProjectCreator(resourceType, resourceType.getName(), null);
			IProject newProject = creator.createJavaProject();
			
			// Set the initial content
			createInitialContent(newProject);
			
			if(monitor.isCanceled()){
				return;
			}
			
			monitor.worked(1);
		}
	}
	
	private void checkIfProjectExistsAndDelete(ResourceType resourceType) throws InvocationTargetException, InterruptedException {
		IProject resourceTypeProject = ResourcesPlugin.getWorkspace().getRoot().getProject(resourceType.determineNameForProject());
		if(resourceTypeProject != null){
			removeExistingProject(resourceTypeProject.getProject());
		}
	}

	private void removeExistingProject(IProject project) throws InvocationTargetException, InterruptedException {
		final IProject projectToDelete = project;
		WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
			protected void execute(IProgressMonitor monitor)
					throws CoreException {
				projectToDelete.delete(true, true, monitor);
			}
		};
			op.run(null);
	}

	private void buildLocationInfo(){
		this.locationInfo = new HashMap<String, String>();
		try {
			ExtendedProperties properties = new ExtendedProperties();
			properties.put(REPOSITORIES, importLocation);

			PluginsCollector collector = new DefaultPluginsCollector();
			collector.configure(properties);
			
			Collection<PluginLocation> pluginLocations = collector.collectPluginLocations();
			
			for (PluginLocation pluginLocation : pluginLocations) {
				File pluginFile = new File(pluginLocation.getManifestLocation().toURI());
				Document pluginDoc = XMLDocumentUtility.loadFile(new FileInputStream(pluginFile),false);
				
				XPath xpath = XPathFactory.newInstance().newXPath();
				String resourceTypeName = (String)xpath.evaluate("extension/parameter[@id='name']/@value", pluginDoc.getDocumentElement(), XPathConstants.STRING);
				if (resourceTypeName != null) {
					this.locationInfo.put(resourceTypeName, pluginLocation.getContextLocation().toURI().getPath());
				}
			}
		} catch (Exception e) {
			throw new RuntimeException("Error while building location info", e);
		}
	}
	
	private String cutPrefix(String fullPath, String prefix){
		int index = fullPath.indexOf(prefix);
		return  fullPath.substring(index + prefix.length());
	}
	
	private void copyProjectFiles(IProject javaProject, String resourceTypeRootPath, List<String> excludes) throws IOException, CoreException{
		File resourceTypeRootFolder = new File(resourceTypeRootPath);
		
		IProject project = javaProject.getProject();
		for (File file : resourceTypeRootFolder.listFiles()) {
			if(!excludes.contains(file.getAbsolutePath())){
				if(file.isDirectory()){
					IFolder destFolder = project.getFolder(cutPrefix(file.getAbsolutePath(), resourceTypeRootFolder.getAbsolutePath()));
					FileUtil.importFilesystemFolderToWorkspace(file, destFolder, true);
				}else if(file.isFile()){
					byte[] contents = FileUtil.readFile(new FileInputStream(file));
					IFile destFile = project.getFile(file.getName());
					ByteArrayInputStream inputStream = new ByteArrayInputStream(contents);
					if (destFile.exists()) {
						destFile.setContents(inputStream, true, false, null);
					} else {
						destFile.create(inputStream, true, null);
					}
				}
			}
		}
	}

	private void createInitialContent(IProject javaProject) throws IOException, CoreException {
		
		// Find the resource type import folder
		String resourceTypeRootPath = this.locationInfo.get(javaProject.getProject().getName());
		File resourceTypeRootFolder = new File(resourceTypeRootPath);
		
		// Imports the files to the project
		List<String> excludes = new ArrayList<String>();
		excludes.add(resourceTypeRootFolder.getAbsolutePath() + File.separator + CLASSES_FOLDER_NAME);
		copyProjectFiles(javaProject, resourceTypeRootPath, excludes);
		
		// Creates a jar file for the "classes" folder content
		createClassesJar(javaProject, resourceTypeRootPath);
	}
	
	private void createClassesJar(IProject javaProject, String resourceTypeRootPath) {
		File sourceFolder = new File(resourceTypeRootPath + File.separator + CLASSES_FOLDER_NAME);

		try {
			if (sourceFolder.exists()) {
				// Prepare the jar entries
				List<String> jarEntries = new ArrayList<String>();
				
				for (File file : sourceFolder.listFiles()) {
					addEntries(jarEntries, file);
				}
				
				if(!jarEntries.isEmpty()){
					// Creates the jar
					ByteArrayOutputStream jarFileOutPut = new ByteArrayOutputStream();
					JarOutputStream jarFileStream = new JarOutputStream(jarFileOutPut);
					
					for (String entry : jarEntries) {
						File source = new File(entry);
						String entryPath = getEntryPath(entry, sourceFolder.getAbsolutePath());
						if(source.isDirectory()){
							jarFileStream.putNextEntry(new JarEntry(entryPath + "/"));
							jarFileStream.closeEntry();
						}else{
							jarFileStream.putNextEntry(new JarEntry(entryPath));
							addJarEntry(new FileInputStream(source), jarFileStream);
						}
					}
					jarFileStream.close();
					jarFileOutPut.close();
					
					IFile jarFile = javaProject.getProject().getFolder(LIB_FOLDER_NAME).getFile(CLASSES_JAR_FILENAME + CLASSES_JAR_EXTENSION);
					// Check if a device.jar file already exists and creates a file with an ordered name 
					int suffix = 1;
					while(jarFile.exists()){
						jarFile = javaProject.getProject().getFolder(LIB_FOLDER_NAME).getFile(CLASSES_JAR_FILENAME + "_" + suffix + CLASSES_JAR_EXTENSION);
						++suffix;
					}
					jarFile.create(new ByteArrayInputStream(jarFileOutPut.toByteArray()), true, null);
				}
			}

		} catch (IOException e) {
			throw new RuntimeException("Error creating jar file for project : " + javaProject.getProject().getName());
		} 
		catch (CoreException e) {
			throw new RuntimeException("Error copying jar file inside project : " + javaProject.getProject().getName());
		}
	}
	
	private void addEntries(List<String> entries, File entry){
		if(entry.isDirectory()){
			if(entry.listFiles().length > 0){
				for (File content : entry.listFiles()) {
					addEntries(entries, content);
				}
			}else{
				entries.add(entry.getPath());
			}
		}else{
			entries.add(entry.getPath());
		}
	}
	
	private void addJarEntry(InputStream in, JarOutputStream out) throws IOException {
		byte[] buffer = new byte[1024];
		
		int len;
		while ((len = in.read(buffer)) >= 0)
			out.write(buffer, 0, len);
		
		out.closeEntry();
		in.close();
	}
	
	private String getEntryPath(String fullPath, String prefix){
		int index = fullPath.indexOf(prefix);
		return  fullPath.substring(index + prefix.length() + 1);
	}
	
	/**
	 * Refreshes the tree views so that newly imported items appear. Yep, this is worse than
	 * police brutality.
	 */
	protected void updateDisplay() {
		/* We are currently in a non-UI thread (this is a job, after all). There won't be an active
		 * workbench window unless we make the call in a UI thread. */
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				refreshViews();
			}
		});
	}
	
	protected void refreshViews() {
		IWorkbenchPage page = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage();
		String [] views = { ID_RESOURCE_GROUP, ID_RESOURCE_TYPE };
		
		for (String viewId : views) {
			IViewPart part =  page.findView(viewId);
		
			if (part != null)
				AbstractTreeViewExplorer.class.cast(part).refresh();
		}
	}
}
