package com.tandbergtv.watchpoint.studio.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.tandbergtv.watchpoint.studio.dto.AdaptorScope;
import com.tandbergtv.watchpoint.studio.dto.AdaptorType;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.JPFConstants;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.JPFExportUtil;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants;

/**
 * Class responsible for generate the plugin.xml document that will be used to create the
 * plugin.xml file in file system.<br>
 * This plugin.xml file represents the Resource type in file system.<br>
 * For now, there is a class similar to this one used by the export mechanism. It is useful to 
 * keep them separated in order to keep the concepts, generate/export, also separated. 
 * Then changes in one mechanism doesn't interferer on the other. 
 * @author xpatrpe
 *
 */
public class ResourceTypePluginFileGenerator {
	// Prefix used when creating the Plug-in Id
	private static final String PLUGIN_ID_PREFIX = "ResourceTypePlugin_";

	// Prefix used when creating the Extension Id
	private static final String EXTENSION_ID_PREFIX = "ResourceType_";

	// Formatter for the Dates in the plug-in document
	private static final String JPF_DATE_FORMAT = "yyyy-MM-dd";
	
	public static Document generatePluginDocument(ResourceType resourceType) throws PluginFileGenerationException {
		
		try {
			// Build the plug-in XML document.
			Document pluginDocument = buildPluginDocument(resourceType);
			
			// Add the Plug-in dependencies
			addPluginDependencies(pluginDocument);
			
			// Add Runtime parameters
			addRuntimeLibraries(pluginDocument);
			
			// Add the Extensions / Extension Points
			addResourceTypeExtension(pluginDocument, resourceType);
			
			return pluginDocument;
			
		} catch (PluginFileGenerationException efe) {
			throw efe;
		} catch (Exception ex) {
			String msg = "Failed to export the WatchPoint Plugin.";
			throw new PluginFileGenerationException(msg, ex);
		}
	}
	
	private static void addRuntimeLibraries(Document document){
		List<String> codeLibraries = new ArrayList<String>();
		codeLibraries.add("bin/");
		List<String> resourceLibraries = new ArrayList<String>();
		resourceLibraries.add("resources/");
		JPFExportUtil.addRuntimeLibraries(document, codeLibraries, resourceLibraries);
	}
	
	private static void addMessagesParameter(Element element, ResourceType resourceType)
	{
		// Add the Messages Element
		Element messagesElement = JPFExportUtil.addParameter(element,
				WatchPointPluginConstants.RT_PARAM_MESSAGES);

		// Add the Default Protocol
		String defaultProtocol = resourceType.getDefaultProtocol();
		if (defaultProtocol == null)
			defaultProtocol = "";

		JPFExportUtil.addParameter(messagesElement,
				WatchPointPluginConstants.RT_PARAM_DEFAULT_PROTOCOL, defaultProtocol);

		// Sort all the Message by UID
		List<Message> messageList = new ArrayList<Message>(resourceType.getMessages());
		Collections.sort(messageList, new Comparator<Message>()
		{
			/**
			 * Sort by UID
			 * 
			 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
			 */
			public int compare(Message o1, Message o2)
			{
				String uid1 = (o1 != null) ? o1.getUid() : "";
				String uid2 = (o2 != null) ? o2.getUid() : "";
				return uid1.compareTo(uid2);
			}
		});

		// Add parameters for each of the sorted messages
		for (Message message : messageList)
		{
			addMessageParameter(messagesElement, message, defaultProtocol);
		}
	}

	/*
	 * Add the 'Message' Parameter for a single Message
	 */
	private static void addMessageParameter(Element element, Message message, String defaultProtocol)
	{
		Element messageElement = JPFExportUtil.addParameter(element,
				WatchPointPluginConstants.RT_PARAM_MESSAGE);

		// Add the Name and UID
		JPFExportUtil.addParameter(messageElement, WatchPointPluginConstants.RT_PARAM_MESSAGE_NAME,
				message.getName());
		JPFExportUtil.addParameter(messageElement, WatchPointPluginConstants.RT_PARAM_MESSAGE_UID,
				message.getUid());

		// Add the Description, if required
		String description = message.getDescription();
		if (description != null && description.trim().length() > 0)
		{
			JPFExportUtil.addParameter(messageElement,
					WatchPointPluginConstants.RT_PARAM_MESSAGE_DESCRIPTION, description);
		}

		// Add the Protocol, if required
		String protocol = message.getProtocol();
		if (protocol != null && protocol.trim().length() > 0 && !protocol.equals(defaultProtocol))
		{
			JPFExportUtil.addParameter(messageElement,
					WatchPointPluginConstants.RT_PARAM_MESSAGE_PROTOCOL, protocol);
		}
	}
	
	private static void addCommunicationParameter(Element element, ResourceType resourceType)
	{
		Set<Message> messages = resourceType.getMessages();
		int messageCount = (messages != null) ? messages.size() : 0;

		// Add the Communication Element
		if(resourceType.getAdaptorType().equals(AdaptorType.NONE)){
			return;
		}
		Element communicationElement = JPFExportUtil.addParameter(element,
				WatchPointPluginConstants.RT_PARAM_COMMUNICATION);
		if (messageCount != 0){
			addMessagesParameter(communicationElement, resourceType);
		}
		if(!resourceType.getAdaptorType().equals(AdaptorType.NONE)){
			addAdaptorParameter(communicationElement, resourceType);
		}
	}
	
	/*
	 * Add the Default Adaptor Parameter
	 */
	private static void addDefaultAdaptorParameter(Element element, ResourceType resourceType)
	{
		Element defaultAdaptorElement = JPFExportUtil.addParameter(element,
				WatchPointPluginConstants.RT_PARAM_ADAPTOR_DEFAULT);

		String messageIdentifier = resourceType.getMessageIdentificationClassName();
		if (messageIdentifier != null && messageIdentifier.trim().length() > 0)
		{
			JPFExportUtil.addParameter(defaultAdaptorElement,
					WatchPointPluginConstants.RT_PARAM_MESSAGE_IDENTIFIER, messageIdentifier);
		}
	}

	/*
	 * Add the Custom Adaptor Parameter
	 */
	private static void addCustomAdaptorParameter(Element element, ResourceType resourceType)
	{
		Element customAdaptorElement = JPFExportUtil.addParameter(element,
				WatchPointPluginConstants.RT_PARAM_ADAPTOR_CUSTOM);

		// Add the custom adaptor class name
		JPFExportUtil.addParameter(customAdaptorElement,
				WatchPointPluginConstants.RT_PARAM_ADAPTOR_CLASS,
				resourceType.getAdaptorClassName());

		// Add the adaptor scope, if required
		String scopeValue = getScopeValue(resourceType.getAdaptorScope());
		if (scopeValue != null)
		{
			JPFExportUtil.addParameter(customAdaptorElement,
					WatchPointPluginConstants.RT_PARAM_ADAPTOR_SCOPE, scopeValue);
		}

		// Add the parameters
		Map<String, String> parameters = resourceType.getAdaptorParameters();
		addResourceTypeParameters(customAdaptorElement, parameters,
				WatchPointPluginConstants.RT_PARAM_ADAPTOR_PARAMETERS,
				WatchPointPluginConstants.RT_PARAM_ADAPTOR_PARAMETER,
				WatchPointPluginConstants.RT_PARAM_ADAPTOR_PARAMETER_KEY,
				WatchPointPluginConstants.RT_PARAM_ADAPTOR_PARAMETER_VALUE);
	}
	
	/*
	 * Add the Adaptor Parameter
	 */
	private static void addAdaptorParameter(Element element, ResourceType resourceType)
	{
		Element adaptorElement = JPFExportUtil.addParameter(element,
				WatchPointPluginConstants.RT_PARAM_ADAPTOR);

		if (resourceType.getAdaptorType() == AdaptorType.DEFAULT)
		{
			addDefaultAdaptorParameter(adaptorElement, resourceType);
		}
		else if (resourceType.getAdaptorType() == AdaptorType.CUSTOM)
		{
			addCustomAdaptorParameter(adaptorElement, resourceType);
		}
	}
	
	private static void addBasicParameters(Element element, ResourceType resourceType)
	{
		// Add the System Id
		JPFExportUtil.addParameter(element, WatchPointPluginConstants.RT_PARAM_SYSTEM_ID,
				resourceType.getSystemId());

		// Add the Name
		JPFExportUtil.addParameter(element, WatchPointPluginConstants.RT_PARAM_NAME,
				resourceType.getName());

		// Add the Create Date
		DateFormat dateFormat = new SimpleDateFormat(JPF_DATE_FORMAT);
		String createDate = dateFormat.format(resourceType.getCreateDate());
		JPFExportUtil.addParameter(element, WatchPointPluginConstants.RT_PARAM_CREATE_DATE,
				createDate);

		// Add the Description, if present
		String description = resourceType.getDescription();
		if (description != null && description.trim().length() > 0)
		{
			JPFExportUtil.addParameter(element, WatchPointPluginConstants.RT_PARAM_DESCRIPTION,
					description);
		}
	}
	
	private static void addResourceTypeExtension(Document document, ResourceType resourceType)
	{
		String extensionId = EXTENSION_ID_PREFIX + resourceType.getSystemId();
		Element extensionElement = JPFExportUtil.addExtension(document, extensionId,
				WatchPointPluginConstants.PLUGIN_ID,
				WatchPointPluginConstants.EXTENSION_POINT_ID_RESOURCE_TYPE);

		addBasicParameters(extensionElement, resourceType);
		addCommunicationParameter(extensionElement, resourceType);
		addManagementParameter(extensionElement, resourceType);
	}
	
	/*
	 * Add the Management Parameter for the Resource Type Plug-in to the input parent element
	 */
	private static void addManagementParameter(Element element, ResourceType resourceType)
	{
		Element managementElement = JPFExportUtil.addParameter(element,
				WatchPointPluginConstants.RT_PARAM_MANAGEMENT);

		// Add the Connection Type
		String connectionType = resourceType.getConnectionType().getName();
		JPFExportUtil.addParameter(managementElement,
				WatchPointPluginConstants.RT_PARAM_CONNECTION_TYPE, connectionType);

		// Add the Initialization Parameters
		addInitializationParameters(managementElement, resourceType);

		// Add the HeartBeat Parameters
		addHeartbeatParameters(managementElement, resourceType);
	}
	
	/*
	 * Adds the list of parameters (alphabetically by parameter key) to the parent element using the
	 * element names provided for the 'parameters' element, the 'parameter' element, the 'key'
	 * element and the 'value' element.
	 */
	private static void addResourceTypeParameters(Element element, Map<String, String> parameters,
			String parametersName, String parameterName, String keyName, String valueName)
	{
		// Check if parameters are required
		if (parameters == null || parameters.size() == 0)
			return;

		// Add the Parameters Element
		Element parametersElement = JPFExportUtil.addParameter(element, parametersName);

		// Sort the parameters alphabetically
		List<String> parameterKeys = new ArrayList<String>(parameters.keySet());
		Collections.sort(parameterKeys);

		// Add all the parameters
		for (String key : parameterKeys)
		{
			String value = parameters.get(key);

			Element parameterElement = JPFExportUtil.addParameter(parametersElement, parameterName);
			JPFExportUtil.addParameter(parameterElement, keyName, key);
			JPFExportUtil.addParameter(parameterElement, valueName, value);
		}
	}

	/*
	 * Add the Initialization Parameter
	 */
	private static void addInitializationParameters(Element element, ResourceType resourceType)
	{
		if (!resourceType.isInitializationRequired())
			return;

		// Add the Initialization Element
		Element initializationElement = JPFExportUtil.addParameter(element,
				WatchPointPluginConstants.RT_PARAM_INITIALIZATION);

		// Add the Strategy Class
		JPFExportUtil.addParameter(initializationElement,
				WatchPointPluginConstants.RT_PARAM_INITIALIZATION_STRATEGY,
				resourceType.getInitializationStrategyClass());

		// Add the Parameters
		Map<String, String> parameters = resourceType.getInitializationParameters();
		addResourceTypeParameters(initializationElement, parameters,
				WatchPointPluginConstants.RT_PARAM_INITIALIZATION_PARAMETERS,
				WatchPointPluginConstants.RT_PARAM_INITIALIZATION_PARAMETER,
				WatchPointPluginConstants.RT_PARAM_INITIALIZATION_PARAMETER_KEY,
				WatchPointPluginConstants.RT_PARAM_INITIALIZATION_PARAMETER_VALUE);
	}

	/*
	 * Add the Heart Beat Parameter
	 */
	private static void addHeartbeatParameters(Element element, ResourceType resourceType)
	{
		if (!resourceType.isHeartbeatRequired())
			return;

		// Add the HeartBeat Element
		Element heartbeatElement = JPFExportUtil.addParameter(element,
				WatchPointPluginConstants.RT_PARAM_HEARTBEAT);

		// Add the Connection Type for heart beats
		JPFExportUtil.addParameter(heartbeatElement,
				WatchPointPluginConstants.RT_PARAM_HEARTBEAT_CONNECTION_TYPE,
				resourceType.getHeartbeatConnectionType().getName());

		// Add the Strategy Class
		JPFExportUtil.addParameter(heartbeatElement,
				WatchPointPluginConstants.RT_PARAM_HEARTBEAT_STRATEGY,
				resourceType.getHeartbeatStrategyClass());

		// Add the Parameters
		Map<String, String> parameters = resourceType.getHeartbeatParameters();
		addResourceTypeParameters(heartbeatElement, parameters,
				WatchPointPluginConstants.RT_PARAM_HEARTBEAT_PARAMETERS,
				WatchPointPluginConstants.RT_PARAM_HEARTBEAT_PARAMETER,
				WatchPointPluginConstants.RT_PARAM_HEARTBEAT_PARAMETER_KEY,
				WatchPointPluginConstants.RT_PARAM_HEARTBEAT_PARAMETER_VALUE);
	}

	protected static Document buildPluginDocument(ResourceType resourceType)
			throws PluginFileGenerationException {
		String pluginId = PLUGIN_ID_PREFIX + resourceType.getSystemId();
		String version = Integer.toString(resourceType.getVersion());

		try {
			return JPFExportUtil.createPluginDocument(pluginId, version);
		} catch (ParserConfigurationException ex) {
			String msg = "Export failed, unable to build the plugin XML document.";
			throw new PluginFileGenerationException(msg, ex);
		}
	}
	
	protected static void addPluginDependencies(Document pluginDocument)
	{
		// Add the dependency to the WatchPoint Plug-in
		JPFExportUtil.addPluginDependency(pluginDocument, WatchPointPluginConstants.PLUGIN_ID);
	}
	
	/**
	 * Write the plug-in document to file 'plugin.xml' in the root folder for the plug-in
	 * 
	 * @param exportable
	 *            The object to export
	 * @param inputs
	 *            The export process inputs
	 * @param pluginDocument
	 *            The plug-in XML Document
	 * @param pluginRootFolder
	 *            The root folder for the plug-in
	 * 
	 * @throws ExportFailureException
	 *             Exception writing the plug-in document
	 */
	public static ByteArrayOutputStream createWritablePluginDocument(Document pluginDocument, 
			File pluginRootFolder) throws PluginFileGenerationException
	{
		ByteArrayOutputStream baos = null;
		
		try
		{
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer();

			// Set the DocType and indentation
			transformer.setOutputProperty(OutputKeys.DOCTYPE_PUBLIC, JPFConstants.DOCTYPE_PUBLIC_ID);
			transformer.setOutputProperty(OutputKeys.DOCTYPE_SYSTEM, JPFConstants.DOCTYPE_SYSTEM_ID);
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
			transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");
			transformer.setOutputProperty("{http://xml.apache.org/xalan}line-separator","\n");
			transformer.setOutputProperty(OutputKeys.STANDALONE, "no");
			
			baos = new ByteArrayOutputStream();
			transformer.transform(new DOMSource(pluginDocument), new StreamResult(baos));
			return baos;
		}
		catch (Exception ex)
		{
			String msg = "Failed to generate the writable Plugin document." ;
			throw new PluginFileGenerationException(msg, ex);
		}
		finally
		{
			if (baos != null)
			{
				try
				{
					baos.close();
				}
				catch (IOException ex)
				{
					String msg = "Failed to close file stream after generate the writable Plugin document. ";
					throw new PluginFileGenerationException(msg, ex);
				}
			}
		}
	}
	
	public static void writePluginFile(Document pluginDocument, String pluginPath) 
						throws PluginFileGenerationException{
		ByteArrayOutputStream baos = null;
		FileOutputStream fos = null;
		try {
			baos = createWritablePluginDocument(pluginDocument, new File(pluginPath));
			fos = new FileOutputStream(pluginPath);
			baos.writeTo(fos);
			fos.flush();
		} catch (Exception e) {
			String msg = "Failed to write the writable Plugin document." ;
			throw new PluginFileGenerationException(msg, e);
		} finally{
			try {
				fos.close();
			} catch (IOException e) {
				String msg = "Failed to close file stream after generate the writable Plugin document. ";
				throw new PluginFileGenerationException(msg, e);
			}
		}
	}
	
	/*
	 * Gets the Value to use for the Adaptor Scope in the Plug-in Document
	 */
	private static String getScopeValue(AdaptorScope scope)
	{
		String scopeValue = null;

		if (scope == null)
			return scopeValue;

		switch (scope)
		{
			case APPLICATION:
				scopeValue = WatchPointPluginConstants.RT_VALUE_ADAPTOR_SCOPE_APPLICATION;
				break;

			case MESSAGE:
				scopeValue = WatchPointPluginConstants.RT_VALUE_ADAPTOR_SCOPE_MESSAGE;
				break;
		}

		return scopeValue;
	}

}
