/*
 * Created on Nov 15, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.actionvariable;

import java.util.ArrayList;
import java.util.List;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;
import com.tandbergtv.watchpoint.studio.validation.model.ActionVariable;

/**
 * Rule that must be subclasses for any validation rule used in the Action Class Configuration. The
 * rule validates that the value is not blank if required. Subclasses may further validate the
 * actual value.
 * 
 * @author Vijay Silva
 */
public class ActionVariableValueRule extends ValidationRule<ActionVariable>
{
	/**
	 * @param target
	 *            The ActionVariable to validate
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(ActionVariable target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		this.validateRequiredValue(target, messages);

		if (!isBlank(target.getElement().getValue()))
			this.validateVariableValue(target, messages);

		return messages;
	}

	/**
	 * validate that the value is present if required.
	 * 
	 * @param target
	 *            the Action variable to validate
	 * @param messages
	 *            The list of validation messages to add to
	 */
	protected void validateRequiredValue(ActionVariable target, List<ValidationMessage> messages)
	{
		boolean valueRequired = (target.getConfiguration() == null) ? false
				: target.getConfiguration().isRequired();
		if (valueRequired && this.isBlank(target.getElement().getValue()))
		{
			ValidationMessageCode code = ValidationMessageCode.ACTION_VARIABLE_VALUE_BLANK;
			List<String> parameters = new ArrayList<String>();
			parameters.add(target.getElement().getName());
			ValidationMessageAdder.getInstance().addValidationMessage(messages, target.getNode(), code, parameters);
		}
	}

	/**
	 * Validate the non-blank Value of the Variable. This method must be overridden to perform
	 * custom validation on the value (eg. ensure it is a number, etc). Ensure that when adding
	 * validation messages to the list, the element in the message must be the node in the Action
	 * Variable.
	 * 
	 * @param target
	 *            The Action Variable to validate
	 * @param messages
	 *            the list of validation messages to add to.
	 */
	protected void validateVariableValue(ActionVariable target, List<ValidationMessage> messages)
	{
	}

	/* Check if the value is blank or null */
	private boolean isBlank(String value)
	{
		return (value == null || value.trim().length() == 0);
	}
}
