package com.tandbergtv.watchpoint.studio.validation.rules.nodeelement;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Validates that all transitions of a NodeElement have a unique non-blank name.
 * 
 * @author Raj Prakash
 * @author Vijay Silva
 */
public class NodeTransitionNamesRule extends ValidationRule<NodeElement> {

	/**
	 * Validates that all outgoing transitions from the given NodeElement have a unique non-blank
	 * name.
	 * 
	 * @param nodeElement
	 *            The Node Element to validate
	 * 
	 * @return The list of validation messages
	 */
	public List<ValidationMessage> validateRule(NodeElement nodeElement) {
		List<ValidationMessage> validationMessages = new ArrayList<ValidationMessage>();

		Transition[] transitions = nodeElement.getTransitions();
		if (transitions != null) {
			Set<String> names = new HashSet<String>();
			Set<String> duplicateNames = new HashSet<String>();
			boolean missingName = false;

			for (Transition transition : transitions) {
				String name = transition.getName();
				boolean isNameBlank = (name == null || name.trim().length() == 0);
				missingName |= isNameBlank;
				if (!isNameBlank) {
					name = name.trim();
					if (names.contains(name))
						duplicateNames.add(name);
					else
						names.add(name);
				}
			}

			this.validateBlankNames(nodeElement, missingName, validationMessages);
			this.validateDuplicateNames(nodeElement, duplicateNames, validationMessages);
		}

		return validationMessages;
	}

	/*
	 * Adds a validation message if any of the Transitions have a blank name
	 */
	private void validateBlankNames(NodeElement node, boolean missing,
			List<ValidationMessage> messages) {
		if (missing) {
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, ValidationMessageCode.NODE_TRANSITIONS_NO_LABEL);
		}
	}

	/*
	 * Adds a validation message for each of the duplicate Transition labels
	 */
	private void validateDuplicateNames(NodeElement node, Set<String> duplicates,
			List<ValidationMessage> messages) {
		for (String name : duplicates) {
			ValidationMessageCode code = ValidationMessageCode.NODE_TRANSITIONS_DUPLICATE_LABEL;
			List<String> parameters = new ArrayList<String>();
			parameters.add(name);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, node, code, parameters);
		}
	}
}
