/**
 * DatatypeConverter.java
 * Created Dec 4, 2006
 * Copyright (C) Tandberg Television 2006
 */
package com.tandbergtv.watchpoint.studio.debugger.core;

import static java.util.Arrays.asList;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;

/**
 * Converts an input value into an Object of the required data type
 * 
 * @author Sahil Verma
 */
public class DatatypeConverter {
	
	/* FIXME Read this from a config file */
	private static final String DATE_FORMAT = "yyyy-MM-dd";
	
	private DateFormat dateformat;

	/**
	 * Creates a DatatypeConverter
	 * @throws Exception 
	 */
	public DatatypeConverter() {
		super();
		this.dateformat = new SimpleDateFormat(DATE_FORMAT); 
	}

	/**
	 * Performs conversion of the input to the right data type. We convert the values for
	 * only certain data types, for the rest (STRING, FILE, DESTINATION etc) we just return the
	 * input converted to a string.
	 * 
	 * @param input
	 * @param type
	 * @return
	 * @throws TypeConversionException if the input can't be converted to the specified data type
	 */
	public Object convert(Object input, DataType type) throws TypeConversionException {
		if (input == null)
			throw new IllegalArgumentException("Input cannot be null");
		
		Object ret = input.toString();
		
		try {
			switch (type) {
				case INT:
					ret = Integer.valueOf(input.toString());
					break;
				case BOOLEAN:
					ret = Boolean.valueOf(input.toString());
					break;
				case DATE:
					ret = dateformat.parse(input.toString());
					break;
				case DURATION:
					/* 
					 * FIXME What now? We could store it as a date, and change the duration
					 * expression parser. Don't know if that's even necessary...
					 */
					break;
				case LIST:
					ret = toList(input);
					break;
				default:
					break;
			}
		} catch (Exception e) {
			throw new TypeConversionException("Failed to convert " + input + " to type " + type, e);
		}
		
		return ret;
	}
	
	/**
	 * Must return type ArrayList because JBPM type conversion relies on that exact type. See
	 * varmapping.xml.
	 * 
	 * @param input
	 * @return
	 */
	private ArrayList<?> toList(Object input) {
		if (input instanceof String) {
			if (input.toString().isEmpty())
				return null; // as opposed to empty list
		
			String [] strings = String.class.cast(input).split(",");
			
			for (int i = 0; i < strings.length; i++)
				strings[i] = strings[i].trim();

			return new ArrayList<String>(asList(strings));
		}
		
		return (ArrayList<?>) input;
	}
}
