package com.tandbergtv.watchpoint.studio.debugger.launching;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationType;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.debug.core.ILaunchManager;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugModelPresentation;
import org.eclipse.debug.ui.ILaunchGroup;
import org.eclipse.debug.ui.ILaunchShortcut2;
import org.eclipse.jdt.launching.IJavaLaunchConfigurationConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.dialogs.ElementListSelectionDialog;

import com.tandbergtv.watchpoint.studio.debugger.WatchpointTemplateDebuggerPlugin;

/**
 * 		Launch shortcut for the Template simulator.
 * 
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio - Francisco Bento da Silva Neto</a>
 *
 */
public class TemplateLaunchShortcut implements ILaunchShortcut2 {

	public void launch(ISelection selection, String mode) {
		launch(getLaunchableResource(selection), mode);
	}

	public void launch(IEditorPart editor, String mode) {
		launch(getLaunchableResource(editor), mode);
	}
	
	protected void launch(IResource resource, String mode) {
		IFile file = (IFile) resource;
		ILaunchConfiguration config = findLaunchConfiguration(file, getConfigurationType());
		if (config == null) {
			// Creates a new configuration
			config = createConfiguration(file);
			ILaunchGroup group = DebugUITools.getLaunchGroup(config, mode);
			// Opens the configuration dialog 
			DebugUITools.openLaunchConfigurationDialog(WatchpointTemplateDebuggerPlugin.getActiveWorkbenchShell(), 
													   config, 
													   group.getIdentifier(), 
													   null);
		} else {
			DebugUITools.launch(config, mode);
		}
	}
	
	/**
	 * 		Creates an empty Template launch configuration.
	 * 
	 * @param file The template file
	 * @return
	 */
	protected ILaunchConfiguration createConfiguration(IFile file) {
		ILaunchConfiguration config = null;
		ILaunchConfigurationWorkingCopy wc = null;
		try {
			ILaunchConfigurationType configType = getConfigurationType();
			String templateName = file.getName().substring(0, file.getName().lastIndexOf("."));
			wc = configType.newInstance(null, getLaunchManager().generateUniqueLaunchConfigurationNameFrom(templateName));
			wc.setAttribute(TemplateLauncherConstants.TEMPLATE_LAUNCHER_PERSISTENT_DATA_TEMPLATE, file.getFullPath().toPortableString());
			wc.setAttribute(IJavaLaunchConfigurationConstants.ATTR_PROJECT_NAME, file.getProject().getName());
			wc.setMappedResources(new IResource[] {file});
			config = wc.doSave();
		} catch (CoreException exception) {
			MessageDialog.openError(WatchpointTemplateDebuggerPlugin.getActiveWorkbenchShell(), "Error creating launch configuration", exception.getStatus().getMessage());	
		} 
		return config;
	}
	
	
	protected ILaunchConfigurationType getConfigurationType() {
		return getLaunchManager().getLaunchConfigurationType(TemplateLauncherConstants.LAUNCH_CONFIGURATION_TYPE_IDENTIFIER);		
	}
	
	/**
	 * Returns the singleton launch manager.
	 * 
	 * @return launch manager
	 */
	private ILaunchManager getLaunchManager() {
		return DebugPlugin.getDefault().getLaunchManager();
	}
	
	/**
	 * Finds and returns an <b>existing</b> configuration to re-launch for the given type,
	 * or <code>null</code> if there is no existing configuration.
	 * 
	 * @return a configuration to use for launching the given type or <code>null</code> if none
	 */
	protected ILaunchConfiguration findLaunchConfiguration(IFile file, ILaunchConfigurationType configType) {
		List<ILaunchConfiguration> candidateConfigs = Collections.emptyList();
		try {
			ILaunchConfiguration[] configs = DebugPlugin.getDefault().getLaunchManager().getLaunchConfigurations(configType);
			candidateConfigs = new ArrayList<ILaunchConfiguration>(configs.length);
			for (int i = 0; i < configs.length; i++) {
				ILaunchConfiguration config = configs[i];
				if (config.getAttribute(TemplateLauncherConstants.TEMPLATE_LAUNCHER_PERSISTENT_DATA_TEMPLATE, "").equals(file.getFullPath().toPortableString())) { //$NON-NLS-1$
					candidateConfigs.add(config);
				}
			}
		} catch (CoreException e) {
		}
		int candidateCount = candidateConfigs.size();
		if (candidateCount == 1) {
			return (ILaunchConfiguration) candidateConfigs.get(0);
		} else if (candidateCount > 1) {
			return chooseConfiguration(candidateConfigs);
		}
		return null;
	}
	
	/**
	 * Returns a configuration from the given collection of configurations that should be launched,
	 * or <code>null</code> to cancel. Default implementation opens a selection dialog that allows
	 * the user to choose one of the specified launch configurations.  Returns the chosen configuration,
	 * or <code>null</code> if the user cancels.
	 * 
	 * @param configList list of configurations to choose from
	 * @return configuration to launch or <code>null</code> to cancel
	 */
	protected ILaunchConfiguration chooseConfiguration(List<ILaunchConfiguration> configList) {
		IDebugModelPresentation labelProvider = DebugUITools.newDebugModelPresentation();
		ElementListSelectionDialog dialog= new ElementListSelectionDialog(WatchpointTemplateDebuggerPlugin.getActiveWorkbenchShell(), labelProvider);
		dialog.setElements(configList.toArray());
		dialog.setTitle("Select Template");  
		dialog.setMessage("Select existing configuration:");
		dialog.setMultipleSelection(false);
		int result = dialog.open();
		labelProvider.dispose();
		if (result == Window.OK) {
			return (ILaunchConfiguration) dialog.getFirstResult();
		}
		return null;		
	}

	public ILaunchConfiguration[] getLaunchConfigurations(ISelection selection) {
		// let the framework resolve configurations based on resource mapping
		return null;
	}

	public ILaunchConfiguration[] getLaunchConfigurations(IEditorPart editorpart) {
		// let the framework resolve configurations based on resource mapping
		return null;
	}

	public IResource getLaunchableResource(ISelection selection) {
		IResource launchable = null;
		if (selection instanceof IStructuredSelection) {
			IStructuredSelection ss = (IStructuredSelection) selection;
			launchable = (IResource) ss.getFirstElement();
		}
		return launchable;
	}

	public IResource getLaunchableResource(IEditorPart editorpart) {
		IResource launchable = null;
		if (editorpart.getEditorInput() instanceof IFileEditorInput) {
			IFileEditorInput in = (IFileEditorInput) editorpart.getEditorInput();
			launchable = (IResource) in.getFile();
		}
		return launchable;
	}
	
}