package com.tandbergtv.watchpoint.studio.debugger.ui;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.forms.widgets.FormToolkit;

import com.tandbergtv.watchpoint.studio.debugger.model.NodeSimulationConfig;
import com.tandbergtv.watchpoint.studio.debugger.model.SimulationType;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;

/**
 * 	Base class for the NodeSimulation Widgets
 * 
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio - Francisco Bento da Silva Neto</a>
 *
 */
public abstract class AbstractNodeSimulationComposite extends Composite implements SelectionListener {

	protected FormToolkit toolkit;
	protected NodeSimulationConfig nodeSimulationConfig;
	protected Label lblNodeType;
	protected Label lblNodeName;
	protected Collection<Button> btnSimulationTypes = new ArrayList<Button>();
	
	protected List<SimulationModifyListener> simulationModifyListeners = new ArrayList<SimulationModifyListener>();
    protected WorkflowTemplate template;
	
	public AbstractNodeSimulationComposite(Composite parent, NodeSimulationConfig nodeConfig) {
		super(parent, SWT.NONE);
		this.nodeSimulationConfig = nodeConfig;
		toolkit = new FormToolkit(parent.getDisplay());
		initializeCompositeData();
		createDefaultControls();
		createAdditionalControls(this);
	}

	/**
	 * 	Initializes the base widget frame.
	 * 
	 */
	protected void initializeCompositeData() {
		toolkit.paintBordersFor(this);
		setLayout(new GridLayout(2, false));
	}
	
	/**
	 * Creates the node name widget
	 * 
	 * @param composite the parent composite
	 * 
	 */
	protected void createNodeNameWidget(Composite composite) {
		Label lblName = new Label(composite, SWT.NONE);
		lblName.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
		lblName.setText("Name :");
		
		lblNodeName = new Label(composite, SWT.NONE);
	}

	/**
	 * Creates the node type widget
	 * 
	 * @param composite the parent composite
	 * 
	 */
	protected void createNodeTypeWidget(Composite composite) {
		Label lblType = new Label(composite, SWT.NONE);
		lblType.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
		lblType.setText("Type :");
		
		lblNodeType = new Label(composite, SWT.NONE);
	}
	
	/**
	 * Creates the simulation type widget
	 * 
	 * @param composite the parent composite
	 * 
	 */
	protected void createSimulationTypeWidget(Composite composite) {
		Label lblSimulationType = new Label(composite, SWT.NONE);
		lblSimulationType.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false, 1, 1));
		lblSimulationType.setText("Simulation Type :");
		
		Composite simulationTypeComposite = toolkit.createComposite(composite, SWT.NONE);
		simulationTypeComposite.setBackground(composite.getBackground());
		toolkit.paintBordersFor(simulationTypeComposite);
		simulationTypeComposite.setLayout(new GridLayout(5, false));
		
		for (SimulationType simulationType : getSimulationTypes()) {
			Button btnSimulationType = new Button(simulationTypeComposite, SWT.RADIO);
			btnSimulationType.setData(simulationType);
			btnSimulationType.addSelectionListener(this);
			btnSimulationType.setText(simulationType.getLabel());
			btnSimulationTypes.add(btnSimulationType);
		}
	}
	
	/**
	 * Creates the default controls
	 */
	protected void createDefaultControls() {
		createNodeNameWidget(this);
		createNodeTypeWidget(this);
		createSimulationTypeWidget(this);
	}

	/**
	 * Creates additional controls. 
	 * 
	 * @param composite The parent composite where the controls should be created
	 */
	protected abstract void createAdditionalControls(Composite composite);
	
	public void load(NodeSimulationConfig nodeSimulationConfig) {
		lblNodeName.setText(nodeSimulationConfig.getNodeElement().getName());
		lblNodeType.setText(nodeSimulationConfig.getNodeElement().getLabel());
		for (Button button : btnSimulationTypes) {
			if (button.getData().equals(nodeSimulationConfig.getSimulationType())) {
				button.setSelection(true);
			} else {
				button.setSelection(false);
			}
		}
	}
	
	/**
	 * 	Subclasses must return the SimulationTypes for this Simulator
	 * 
	 * @return
	 */
	protected abstract SimulationType[] getSimulationTypes();

	/**
	 * 		Notifies when a widget is selected. Subclasses may override.
	 */
	public void widgetSelected(SelectionEvent e) {
	}
	
	/**
	 * 		Notifies when the default widget is selected. Subclasses may override.
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	
	public NodeSimulationConfig getNodeSimulationConfig(NodeSimulationConfig config) {
		return config; 
	}

	public void addSimulationModifyListener(SimulationModifyListener listener) {
		simulationModifyListeners.add(listener);
	}

	protected void fireSimulationModifiedEvent(Object o) {
		for (SimulationModifyListener listener : simulationModifyListeners) {
			listener.modifySimulation(o);
		}
	}

    public void setTemplate(WorkflowTemplate template) {
        this.template = template;
    }
}
