/*
 * Created on Jun 26, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.dataaccess;

import java.io.Serializable;

import com.tandbergtv.watchpoint.studio.dto.IPersistable;
import com.tandbergtv.watchpoint.studio.util.ClassLoaderUtil;

/**
 * Abstract Factory for instantiating Data Access objects.
 * 
 * @author Vijay Silva
 */
public abstract class DataAccessFactory
{
	private static final String DEFAULT_FACTORY = "com.tandbergtv.watchpoint.studio.dataaccess.jpa.JPADataAccessFactory";

	/**
	 * Constructor
	 */
	protected DataAccessFactory()
	{
		super();
	}

	// ========================================================================
	// ==================== FACTORY CREATION
	// ========================================================================

	/**
	 * Return the configured Data Access Factory to use for creating Data Access Objects.
	 * 
	 * @return The DataAccessFactory
	 */
	public static DataAccessFactory createFactory()
	{
		return ClassLoaderUtil.createFactoryInstance(DEFAULT_FACTORY, DataAccessFactory.class);
	}

	// ========================================================================
	// ==================== GENERIC OBJECT CREATION
	// ========================================================================

	/**
	 * Create a Data Access Object that can be used for the sample entity.
	 * 
	 * @param <DO>
	 *            The type of the entity, which must be persistable
	 * @param <DOK>
	 *            The type of the entity key
	 * @param entity
	 *            The entity (or a sample entity) for which data operations need to be performed.
	 * @param context
	 *            The persistence context to use for the data operations.
	 * 
	 * @return The Data Access Object
	 */
	public abstract <DO extends IPersistable, DOK extends Serializable> DataAccessInterface<DO, DOK> createDataAccessObject(
			DO entity, IPersistenceContext context);

	/**
	 * Create a Data Access Object that can be used for operations dealing with the entity class.
	 * 
	 * @param <DO>
	 *            The type of the entity, which must be persistable
	 * @param <DOK>
	 *            The type of the entity key
	 * @param entityClass
	 *            The Class of the entity for which data operations need to be performed.
	 * @param context
	 *            The persistence context to use for the data operations.
	 * 
	 * @return The Data Access Object
	 */
	public abstract <DO extends IPersistable, DOK extends Serializable> DataAccessInterface<DO, DOK> createDataAccessObject(
			Class<DO> entityClass, IPersistenceContext context);

	// ========================================================================
	// ==================== SPECIFIC OBJECT CREATION
	// ========================================================================

	/**
	 * Create a Data Access Object for the ResourceType entity
	 * 
	 * @param context
	 *            The Persistence context to use for data operations.
	 * 
	 * @return the ResourceTypeDAI implementation
	 */
	public abstract ResourceTypeDAI getResourceTypeDAO(IPersistenceContext context);

	/**
	 * Create a Data Access Object for the Message entity
	 * 
	 * @param context
	 *            The Persistence context to use for data operations.
	 * 
	 * @return the MessageDAI implementation
	 */
	public abstract MessageDAI getMessageDAO(IPersistenceContext context);

	/**
	 * Create a Data Access Object for the ResourceGroup entity
	 * 
	 * @param context
	 *            The Persistence context to use for data operations.
	 * 
	 * @return the ResourceGroupDAI implementation
	 */
	public abstract ResourceGroupDAI getResourceGroupDAO(IPersistenceContext context);

	/**
	 * Create a Data Access Object for the ProtectionKey entity
	 * 
	 * @param context
	 *            The Persistence context to use for data operations.
	 * 
	 * @return the ProtectionKeyDAI implementation
	 */
	public abstract ProtectionKeyDAI getProtectionKeyDAO(IPersistenceContext context);

	/**
	 * Create a Data Access Object for the NodeDefinitionDTO entity
	 * 
	 * @param context
	 *            The Persistence context to use for data operations.
	 * 
	 * @return the NodeDefinitionDTODAI implementation
	 */
	public abstract NodeDefinitionDTODAI getNodeDefinitionDTODAO(IPersistenceContext context);

	/**
	 * Create a Data Access Object for the WorkflowTemplateDTO entity
	 * 
	 * @param context
	 *            The Persistence context to use for data operations.
	 * 
	 * @return the WorkflowTemplateDTODAI implementation
	 */
	public abstract WorkflowTemplateDTODAI getWorkflowTemplateDTODAO(IPersistenceContext context);
}