/*
 * Created on Jul 31, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.external.wpexport.impl;

import static com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants.TEMPLATE_SEID;
import static com.tandbergtv.watchpoint.studio.util.SemanticElementUtil.TRANSITION_PATH_SEPARATOR;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.jbpm.gd.common.model.SemanticElementFactory;
import org.jbpm.gd.jpdl.model.AbstractNode;
import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.model.ActionElement;
import org.jbpm.gd.jpdl.model.Decision;
import org.jbpm.gd.jpdl.model.EndState;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;
import org.jbpm.gd.jpdl.model.ProcessDefinition;
import org.jbpm.gd.jpdl.model.SuperState;
import org.jbpm.gd.jpdl.model.Transition;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.external.wpexport.IWatchPointDTOExporter;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.template.ActionInserter;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.template.ExpressionsFixer;
import com.tandbergtv.watchpoint.studio.external.wpexport.impl.template.UniqueNodeNameGenerator;
import com.tandbergtv.watchpoint.studio.ui.model.AutomaticTaskNode;
import com.tandbergtv.watchpoint.studio.ui.model.LoopNode;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.SemanticElementConstants;
import com.tandbergtv.watchpoint.studio.ui.model.WPTransition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.util.DecisionExprValidationUtil;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

/**
 * Exporter for a Workflow Template that writes the XML that needs to be exported to the file system
 * using a filepath specified in the inputs.
 * 
 * @author Vijay Silva
 */
public abstract class WorkflowTemplateExporter implements IWatchPointDTOExporter<WorkflowTemplateDTO> {

	private static final String PARENT = "..";

	
	private static final Lock MODEL_LOCK = new ReentrantLock();
	
	/**
	 * Tweaks the UI model in order to make it ready for export as a valid JPDL
	 * 
	 * @param template
	 */
	protected void postProcess(WorkflowTemplate template) {
		this.convertExpression(template);
		this.checkAndFixSuperStateNodeNames(template);
		ActionInserter.addActions(template);
		this.removeNodeDefinitions(template);
		this.removeLoopNodeType(template);
	}

	// ========================================================================
	// ===================== WORKFLOW TEMPLATE CREATION
	// ========================================================================

	/*
	 * Create a WorkflowTemplate object from the Xml String.
	 */
	protected Map<String, Object> createWorkflowTemplate(String xml) throws ExportFailureException {
		MODEL_LOCK.lock();

		try {
			return SemanticElementUtil.createSemanticElement(xml, TEMPLATE_SEID);
		}
		catch (Exception ex) {
			String msg = "Failed to construct Semantic Element: WorkflowTemplate from the XML String.";
			throw new ExportFailureException(msg, ex);
		}
		finally {
			MODEL_LOCK.unlock();
		}
	}

	// ========================================================================
	// ===================== EXPRESSION HANDLING
	// ========================================================================

	/*
	 * Converts the expression in all the node contained in the parent container.
	 */
	private void convertExpression(NodeElementContainer parent) {
		ExpressionsFixer.fixExpressions(parent);
	}

	// ========================================================================
	// ===================== SUPER STATE NODE NAME HANDLING
	// ========================================================================

	/*
	 * Ensures that all nodes within a super state have unique names that don't clash with the
	 * template's top level nodes.
	 */
	private void checkAndFixSuperStateNodeNames(WorkflowTemplate template) {
		UniqueNodeNameGenerator nameHelper = new UniqueNodeNameGenerator(true);
		nameHelper.setRootContainer(template);
		nameHelper.fixSuperStateNodeNames();
	}

	// ========================================================================
	// ===================== NODE DEFINITION HANDLING
	// ========================================================================

	/*
	 * Removes all Node Definitions contained in the parent Node Container's list of child nodes.
	 */
	private void removeNodeDefinitions(NodeElementContainer parent) {
		List<NodeElement> orderedNodes = new ArrayList<NodeElement>();
		boolean requiresReordering = false;

		NodeElement[] childNodes = parent.getNodeElements();
		if (childNodes == null) { // Check if no child nodes are present
			return;
		}

		for (NodeElement node : childNodes) {
			if (node instanceof NodeDefinition) {
				NodeDefinition nodeDefinition = (NodeDefinition) node;

				/* Remove all unmapped variables in the node definition */
				this.removeUnmappedNodeDefinitionVariables(nodeDefinition);

				/* Need to replace Node Definition Element with contained node/s */
				orderedNodes.add(nodeDefinition.getNode());
				requiresReordering = true;

				switch (nodeDefinition.getNodeType()) {
					case MessageNode:
						this.fixSingleNodeNodeDefinition(parent, nodeDefinition);
						break;

					case SuperState:
						this.fixSuperStateNodeDefinition(parent, nodeDefinition);
						break;
				}
			} else if (node instanceof LoopNode) {
				removeNodeDefinitions((NodeElementContainer) node);
				orderedNodes.add(node);
			} else {
				orderedNodes.add(node);
			}
		}

		/* Remove unwanted Node Definitions and add the contained nodes */
		if (requiresReordering) {
			// Remove all nodes and add from the ordered node list again to maintain node order.
			for (NodeElement node : parent.getNodeElements())
				parent.removeNodeElement(node);

			for (NodeElement node : orderedNodes)
				parent.addNodeElement(node);
		}
	}
	
	/**
	 * Removes the type attribute from the loop node. The type is important only in the jpdl file 
	 * @param parent The container holding the loop node
	 */
	private void removeLoopNodeType(NodeElementContainer parent) {
		NodeElement[] childNodes = parent.getNodeElements();
		if (childNodes == null) { // Check if no child nodes are present
			return;
		}
		for (NodeElement node : childNodes) {
			if( node instanceof LoopNode ){
				((LoopNode)node).setLoopType(null);
			} else if( node instanceof NodeGroup ){
				removeLoopNodeType((NodeGroup)node);
			}
		}
	}

	/*
	 * Removes all occurrences of the variable from the Node Definition if unmapped
	 */
	private void removeUnmappedNodeDefinitionVariables(NodeDefinition nodeDefinition) {
		List<WPVariable> nodeDefinitionVariables = new ArrayList<WPVariable>();
		if (nodeDefinition.getVariables() != null)
			nodeDefinitionVariables.addAll(nodeDefinition.getVariables());

		for (WPVariable variable : nodeDefinitionVariables) {
			String processVariableName = (variable != null) ? variable.getName() : null;
			if (processVariableName == null || processVariableName.trim().length() == 0)
				this.removeUnmappedNodeDefinitionVariable(nodeDefinition, variable);
		}
	}

	/*
	 * Given the Node Definition Variable that is not mapped, remove the variable and its uses from
	 * the Node Definition.
	 */
	private void removeUnmappedNodeDefinitionVariable(NodeDefinition nodeDefinition,
			WPVariable variable) {
		NodeDefinitionType type = nodeDefinition.getNodeType();

		/* Handle Single-Node Node Definitions */
		if (type == NodeDefinitionType.MessageNode) {
			/* Remove the variable from the contained Automatic Task Node */
			AutomaticTaskNode node = (AutomaticTaskNode) nodeDefinition.getNode();
			node.removeVariable(variable.getMappedName());
		}

		/* Handle SuperState Node Definitions */
		else if (type == NodeDefinitionType.SuperState) {
			/* Remove Super State variable and remove all child node variables mapped to it */
			String superStateVariableName = variable.getMappedName();
			nodeDefinition.removeVariable(superStateVariableName);

			NodeElement[] nodeElements = nodeDefinition.getNodeElements();
			if (nodeElements == null)
				return;

			for (NodeElement nodeElement : nodeElements) {
				if (nodeElement instanceof NodeDefinition) {
					NodeDefinition innerNodeDefinition = (NodeDefinition) nodeElement;
					List<WPVariable> innerVariables = innerNodeDefinition.getVariables();
					if (innerVariables == null)
						continue;

					AutomaticTaskNode node = (AutomaticTaskNode) innerNodeDefinition.getNode();
					for (WPVariable innerVariable : innerVariables) {
						if (superStateVariableName.equals(innerVariable.getName()))
							node.removeVariable(innerVariable.getMappedName());
					}
				}
			}
		}
	}

	// ========================================================================
	// ===================== NODE DEFINITION HELPER METHODS
	// ========================================================================

	/*
	 * Update all incoming / outgoing Transitions of the Node Definition to instead link to the
	 * contained single node.
	 */
	private void fixSingleNodeNodeDefinition(NodeElementContainer parent,
			NodeDefinition childNodeDefinition) {
		AbstractNode containedNode = childNodeDefinition.getNode();

		/*
		 * Since Node Definition Name and contained Node Name are the same, do nothing for incoming
		 * transitions (which use node name for destination). Move outgoing transitions from the
		 * Node Definition to the contained node.
		 */
		Transition[] transitions = childNodeDefinition.getTransitions();
		for (Transition transition : transitions) { // Move the transition to the contained node
			childNodeDefinition.removeTransition(transition);

			transition.setSource(containedNode);
			
			if (containedNode != null)
				containedNode.addTransition(transition);
		}
	}

	/*
	 * Replace the Node Definition in the parent with the super state contained in the Node
	 * Definition. Ensure that all incoming and outgoing transitions of the SuperState are 'fixed'.
	 */
	private void fixSuperStateNodeDefinition(NodeElementContainer parent,
			NodeDefinition childNodeDefinition) {
		SuperState superState = (SuperState) childNodeDefinition.getNode();

		/* Removes node definition variables */
		this.removeNodeDefinitionVariables(childNodeDefinition, childNodeDefinition.getVariables());

		/* Remove nested Node Definitions */
		this.removeNodeDefinitions(superState);

		/* Get 'Entry' node of Super State (only one) and update the incoming transitions */
		NodeElement entryNode = this.getSuperStateEntryNode(superState);
		String transitionPathToAppend = TRANSITION_PATH_SEPARATOR + entryNode.getName();
		Set<Transition> inTransitions = this.getIncomingTransitions(parent, childNodeDefinition);
		for (Transition transition : inTransitions) {
			transition.setTo(transition.getTo() + transitionPathToAppend);
		}

		/* Get 'Exit' node of the Super State (only one) and update the outgoing transitions */
		EndState exitNode = this.getSuperStateExitNode(superState);
		
		/* Get all the transitions going to the exit node */
		Set<Transition> exitTransitions = this.getTransitionsToNode(superState, exitNode);
		
		if (childNodeDefinition.getTransitions().length > 0) {
			WPTransition nodeDefExitTransition = (WPTransition) childNodeDefinition.getTransitions()[0];
			
			/* Remove outgoing transition from the node definition */
			childNodeDefinition.removeTransition(nodeDefExitTransition);
			
			/*
			 * Connect super-state exit transitions to the destination of the node definition outgoing
			 * transition. Also add action elements.
			 */
			for (Transition trans : exitTransitions) {
				WPTransition transition = (WPTransition) trans;
				String prefix = "";
				
				/* Add prefix (go up one level) to path if our parent isn't root */
				if (!(parent instanceof ProcessDefinition))
					prefix = PARENT;
				
				transition.setTo(prefix + TRANSITION_PATH_SEPARATOR + nodeDefExitTransition.getTo());
				
				for (ActionElement action : nodeDefExitTransition.getActionElements()) {
					if (action instanceof Action) {
						Action newAction = cloneAction((Action) action, trans.getFactory());
						this.addRefAction(transition, newAction);
					}
				}
			}
		} else {
			if (parent instanceof LoopNode) {
				/* This superstate has no outbound transitions because it's the last node of a parent LoopNode,
				 * 	we can simply get rid of the transitions to the fake End node */
				for (NodeElement node : superState.getNodeElements()) {
					if (node.getTransitions().length > 0) {
						for (Transition t : node.getTransitions()) {
							if (exitTransitions.contains(t))
								node.removeTransition(t);
						}
					}
				}
			}
		}

		/* Remove the Exit node from the Super State, move all transitions from the exit node */
		superState.removeNodeElement(exitNode);
	}

	/*
	 * Maps template variables to nested node definition variables in a super-state node definition
	 * using its variables
	 */
	private void removeNodeDefinitionVariables(NodeElementContainer container, List<WPVariable> nodeDefinitionVars) {
		for (NodeElement nodeElement : container.getNodeElements()) {
			if (nodeElement instanceof NodeDefinition) {
				NodeDefinition nestedNodeDefinition = (NodeDefinition) nodeElement;
				for (WPVariable var : nestedNodeDefinition.getVariables()) {
					WPVariable nodeDefVar = this.findVariableByMappedName(nodeDefinitionVars, var
							.getName());
					/* The variable could be a composite key */
					if (nestedNodeDefinition.getCompositeKey(var.getMappedName()) != null)
						nestedNodeDefinition.setCompositeKey(var.getMappedName(), nodeDefVar
								.getName());

					var.setName(nodeDefVar.getName());
				}
			}
			else if (nodeElement instanceof Decision) {
				Decision decision = (Decision) nodeElement;
				String expression = decision.getExpression();
				expression = createRemovedDefinitionVariablesExpression(expression, nodeDefinitionVars);
				decision.setExpression(expression);
			}
			else if (nodeElement instanceof LoopNode) {
				LoopNode loopNode = (LoopNode) nodeElement;
				
				/* Fix the index variable */
				String index = loopNode.getIndex();
				WPVariable v = findVariableByMappedName(nodeDefinitionVars, index);
				
				loopNode.setIndex(v.getName());
				
				String name = loopNode.getListName();
				
				if (name != null) {
					WPVariable listvar = findVariableByMappedName(nodeDefinitionVars, name);
					
					loopNode.setListName(listvar.getName());
				}
				
				/* Fix the expression condition */
				String expression = loopNode.getExpression();
				
				expression = createRemovedDefinitionVariablesExpression(expression, nodeDefinitionVars);
				loopNode.setExpression(expression);
				
				removeNodeDefinitionVariables(loopNode, nodeDefinitionVars);
			}
		}
	}
	
	private String createRemovedDefinitionVariablesExpression(String expression, List<WPVariable> nodeDefinitionVars) {
		if (expression != null && expression.length() > 0) {
			Map<Integer, String> operandMap = DecisionExprValidationUtil.getOperands(expression);
			Integer[] indexArray = operandMap.keySet().toArray(new Integer[0]);
			Arrays.sort(indexArray);
			int exprLength = expression.length();
			
			for (Integer i : indexArray) {
				int index = i.intValue();
				String operand = operandMap.get(index);
				WPVariable nodeDefVar = this.findVariableByMappedName(nodeDefinitionVars, operand);
				/*
				 * We need to keep updating the index since it changes each time we replace
				 * operands
				 */
				int newIndex = index - (exprLength - expression.length());
				String prefix = expression.substring(0, newIndex);
				String suffix = expression.substring(newIndex + operand.length());
				expression = prefix + nodeDefVar.getName() + suffix;
			}
		}
		
		return expression;
	}

	/*
	 * Finds a variable in a given list by its mapped name.
	 */
	private WPVariable findVariableByMappedName(List<WPVariable> variables, String mappedName) {
		for (WPVariable var : variables) {
			if (var.getMappedName().equals(mappedName))
				return var;
		}

		return null;
	}

	// ========================================================================
	// ===================== SUPER STATE HELPER METHODS
	// ========================================================================

	/*
	 * Find the Entry Node for the Super State. The Entry Node is the node in the super state with
	 * no incoming transitions. There should be exactly one such node in a valid super state.
	 * Assumes that the Node Definition is valid.
	 */
	private NodeElement getSuperStateEntryNode(SuperState superState) {
		Set<String> nodesWithIncomingTransition = new HashSet<String>();

		// Build a list of all node names that have an incoming transition
		for (NodeElement node : superState.getNodeElements()) {
			for (Transition transition : node.getTransitions()) {
				nodesWithIncomingTransition.add(transition.getTo());
			}
		}

		// Find the first node without an incoming transition (only one such node should exist)
		NodeElement entryNode = null;
		for (NodeElement node : superState.getNodeElements()) {
			if (!nodesWithIncomingTransition.contains(node.getName())) {
				entryNode = node;
				break;
			}
		}

		return entryNode;
	}

	/*
	 * Find the Exit Node for the Super State. The Exit Node is the node in the super state with no
	 * outgoing transitions. There should be exactly one such node in a valid super state. The Exit
	 * Node should also be of type SuperState End State. Assumes that the Node Definition is valid.
	 */
	private EndState getSuperStateExitNode(NodeElementContainer container) {
		for (NodeElement node : container.getNodeElements()) {
			if (node instanceof EndState)
				return (EndState) node;
		}

		throw new IllegalArgumentException(((AbstractNode) container).getName() + " does not have an End node");
	}

	/*
	 * Gets a set of transitions in the super state going to the specified endstate
	 */
	private Set<Transition> getTransitionsToNode(NodeElementContainer container, EndState node) {
		Set<Transition> transitions = new HashSet<Transition>();
		
		for (NodeElement currentNode : container.getNodeElements()) {
			for (Transition transition : currentNode.getTransitions()) {
				if (transition.getTo().equals(node.getName()))
					transitions.add(transition);
			}
		}
		
		return transitions;
	}

	/*
	 * Find all the incoming transitions for a given Super State Node Definition
	 */
	private Set<Transition> getIncomingTransitions(NodeElementContainer template,
			NodeDefinition nodeDefinition) {
		Set<Transition> incomingTransitions = new HashSet<Transition>();

		for (NodeElement node : template.getNodeElements()) {
			this.getIncomingTransitions(node, nodeDefinition, false, incomingTransitions);

			/* Possible that the transition has been moved to within the Node Definition */
			if (node instanceof NodeDefinition) {
				NodeDefinition currentNodeDefinition = (NodeDefinition) node;
				AbstractNode containedNode = currentNodeDefinition.getNode();
				switch (currentNodeDefinition.getNodeType()) {
					case MessageNode:
						this.getIncomingTransitions(containedNode, nodeDefinition, false,
								incomingTransitions);
						break;

					case SuperState:
						SuperState containedSuperState = (SuperState) containedNode;
						for (NodeElement superStateNode : containedSuperState.getNodeElements()) {
							this.getIncomingTransitions(superStateNode, nodeDefinition, true,
									incomingTransitions);
						}
						break;
				}
			}
		}

		return incomingTransitions;
	}

	/*
	 * Finds all the incoming transitions from the 'source' node to the 'target' node definition,
	 * and stores the transitions in the incomingTransitions set. 'isSuperState' indicates that the
	 * source node is part of a super state.
	 */
	private void getIncomingTransitions(NodeElement source, NodeDefinition target,
			boolean isSuperStateNode, final Set<Transition> incomingTransitions) {
	    if (source == null || source.getTransitions() == null) {
	        return;
	    }
		/* Find all incoming transitions to the Node Definition */
		for (Transition transition : source.getTransitions()) {
			if (this.isIncomingTransition(transition, target, isSuperStateNode)) {
				// Found an incoming transition
				incomingTransitions.add(transition);
			}
		}
	}

	/*
	 * Checks if the Transition is an incoming transition for the node definition
	 */
	private boolean isIncomingTransition(Transition transition, NodeElement node,
			boolean fromSuperStateNode) {
		String dest = transition.getTo();
		String name = node.getName();
		String pathQualifiedName = TRANSITION_PATH_SEPARATOR + name;
		
		//If the transition lies within a loop node, need to use the relative path (../parent/child)
		String relativePathName = PARENT + pathQualifiedName;

		return (fromSuperStateNode) ? dest.equals(pathQualifiedName) || dest.equals(relativePathName) : dest.equals(name);
	}

	// ========================================================================
	// ===================== ACTION HELPER METHODS
	// ========================================================================

	/*
	 * Adds an action to the transition with the given ref name, if action doesn't have ref name.
	 */
	private void addRefAction(WPTransition transition, Action action) {
		if (!transition.hasAction(action.getRefName()))
			transition.addActionElement(action);
	}

	/*
	 * Creates a clone of the action passed.
	 */
	private Action cloneAction(Action action, SemanticElementFactory factory) {
		Action newAction = this.createAction(factory);
		newAction.setAsync(action.getAsync());
		newAction.setClassName(action.getClassName());
		newAction.setAcceptPropagatedEvents(action.getAcceptPropagatedEvents());
		newAction.setConfigInfo(action.getConfigInfo());
		newAction.setClassName(action.getClassName());
		newAction.setConfigType(action.getConfigType());
		newAction.setExpression(action.getExpression());
		newAction.setName(action.getName());
		newAction.setRefName(action.getRefName());
		return newAction;
	}

	/*
	 * Creates a new instance of Action
	 */
	private Action createAction(SemanticElementFactory factory) {
		return (Action) factory.createById(SemanticElementConstants.ACTION_SEID);
	}
}
