/*
 * Created on Jul 5, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.service;

import java.util.List;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;

/**
 * Service interface defining operations on NodeDefinitionDTO entities.
 * 
 * @author Vijay Silva
 */
public interface INodeDefinitionService extends IService {

	/**
	 * Method to get the list of all available Node Definitions in the system.
	 * 
	 * @return The list of matching Node Definitions
	 */
	List<NodeDefinitionDTO> getAllNodeDefinitions();

	/**
	 * Method to get the list Node Definitions for a Resource Type.
	 * 
	 * @param resourceTypeId
	 *            The Resource Type Id
	 * 
	 * @return The list of matching Node Definitions
	 */
	List<NodeDefinitionDTO> getNodeDefinitionsByResourceType(long resourceTypeId);
	
	/**
	 * Get a Node Definition DTO by its path. 
	 * 
	 * @param path
	 *            The Node Definition DTO path.
	 * 
	 * @return The NodeDefinitionDTO.
	 */
	NodeDefinitionDTO getNodeDefinitionDTOByPath(String path);
	
	/**
	 * Method to get the list Node Definitions for a Message.
	 * 
	 * @param messageUID
	 *            The Message UID
	 * @param type 
	 * 			  The NodeDefinitionType to filter. 
	 * 
	 * @return The list of matching Node Definitions
	 */
	List<NodeDefinitionDTO> getNodeDefinitionsByMessageUID(String messageUID, NodeDefinitionType type);
	
	/**
	 * Updates a given Node Definition in file system
	 * @param nodeDTO The Node Definition to be updated
	 */
	public void updateNodeDefinitionInFileSystem(NodeDefinitionDTO nodeDTO);
	
	/**
	 * Deletes a given Node Definition in file system
	 * @param nodeDTO The Node Definition to be deleted
	 * @param removeReferences A flag to indicate if references can be deleted without ask the user
	 */
	public void deleteNodeDefinitionInFileSystem(NodeDefinitionDTO nodeDTO, boolean removeReferences);
	
	/**
	 * Creates a Node Definition in file system
	 * @param nodeDTO The node definition to be created
	 */
	public void createNodeDefinitionInFileSystem(NodeDefinitionDTO nodeDTO);
	
	/**
	 * Method to get the list Node Definitions of type Single Node for a Resource Type.
	 * 
	 * @param resourceTypeId
	 *            The Resource Type Id
	 * 
	 * @return The list of matching Node Definitions
	 */
	List<NodeDefinitionDTO> getSingleNodeDefinitionsByResourceTypeAndMessageType(long resourceTypeId, boolean incoming);

	/**
	 * Method to get a Node Definition given the Node Definition Id.
	 * @param nodeDefinitionId The Node Definition Id.
	 * @return The matching Node Definition
	 */
	NodeDefinitionDTO getNodeDefinition(long nodeDefinitionId);

	/**
	 * Creates a new Node Definition.
	 * 
	 * @param nodeDefinition
	 *            The Node Definition
	 * @param userId
	 *            The Id of the user creating the node definition
	 * 
	 * @return The created Node Definition
	 */
	NodeDefinitionDTO createNodeDefinition(NodeDefinitionDTO nodeDefinition);

	/**
	 * Delete a Node Definition by id.
	 * 
	 * @param nodeDefinitionId
	 *            The Id of the Node Definition to delete.
	 * @param removeReferences
	 *            Flag to indicate whether references to this node definition need to be deleted
	 */
	void deleteNodeDefinition(long nodeDefinitionId, boolean removeReferences);

	/**
	 * Checks to see if a node definition with this name already exists in the database.
	 * 
	 * @param name
	 *            the name to check against.
	 * @return true if a node definition with this name exists in the database, false otherwise
	 */
	boolean nameExists(String name);

	/**
	 * Checks if the Node Definition is currently used by existing Node Definitions
	 * 
	 * @param nodeDefinitionId
	 *            The Node Definition Id
	 * @return true if used by any existing node definition, false otherwise
	 */
	boolean isUsedByNodeDefinition(long nodeDefinitionId);
}
