package com.tandbergtv.watchpoint.studio.ui.editor.expression;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.eclipse.core.resources.IProject;
import org.eclipse.jdt.core.compiler.IProblem;
import org.eclipse.jface.bindings.keys.KeyStroke;
import org.eclipse.jface.bindings.keys.ParseException;
import org.eclipse.jface.fieldassist.ComboContentAdapter;
import org.eclipse.jface.fieldassist.ContentProposalAdapter;
import org.eclipse.jface.fieldassist.IControlContentAdapter;
import org.eclipse.jface.fieldassist.TextContentAdapter;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.model.IExpression;
import com.tandbergtv.watchpoint.studio.ui.model.IValidatableElement;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.properties.DataType;
import com.tandbergtv.watchpoint.studio.validation.rules.expression.VariableExpressionValidator;

/**
 * 		Adds code complete support for TextFields and Combos whose have expression support (BeanShell).
 * 
 * 
 * @author <a href="mailto:francisco.bento.silva.neto@ericsson.com">efrasio - Francisco Bento da Silva Neto</a>
 *
 */
public class ExpressionEditor {

    private static KeyStroke keyStroke;
    static {
        try {
            keyStroke = KeyStroke.getInstance("Ctrl+Space");
        } catch (ParseException e) {
        }
    }

    private TableViewer tableViewer;
    private int columnIndex;
    private IExpression expression;
    protected Collection<? extends Variable> contextVariables;
    protected IProject project;
    protected DataType returnType = DataType.OBJECT;

    protected IControlContentAdapter contentAdapter;
    private ContentProposalAdapter proposalAdapter;
    private SnippetCompletionProcessor snippetCompletionProcessor = new SnippetCompletionProcessor();
    private VariableExpressionValidator validator = new VariableExpressionValidator();

    public ExpressionEditor(TableViewer tableViewer, int columnIndex) {
        this.tableViewer = tableViewer;
        this.columnIndex = columnIndex;
        new ExpressionEditorDefaultTooltip(tableViewer);
    }

    public void initialize() {
        validator.setProject(project);
        validator.setContextVariables(ExpressionUtil.toJavaVariableStr(contextVariables));

        // add focus behavior
        (tableViewer.getCellEditors()[columnIndex]).getControl().addFocusListener(focusListener);

        // calculates content adapter
        if ((tableViewer.getCellEditors()[columnIndex]).getControl() instanceof Text) {
            contentAdapter = new TextContentAdapter();
        } else if ((tableViewer.getCellEditors()[columnIndex]).getControl() instanceof CCombo) {
            contentAdapter = new CComboContentAdapter();
        } else if ((tableViewer.getCellEditors()[columnIndex]).getControl() instanceof Combo) {
            contentAdapter = new ComboContentAdapter();
        }

        // effectively add the proposal adapter
        snippetCompletionProcessor.setProject(project);
        snippetCompletionProcessor.setContextVars(ExpressionUtil.toJavaVariableStr(contextVariables));
        proposalAdapter = new ContentProposalAdapter((tableViewer.getCellEditors()[columnIndex]).getControl(),
                contentAdapter, snippetCompletionProcessor, keyStroke, new char[] { '(', '.' });
        proposalAdapter.setProposalAcceptanceStyle(ContentProposalAdapter.PROPOSAL_REPLACE);
        proposalAdapter.setEnabled(false);

        // validate all cells
        validateAllWidgets();
    }

    private void validateAllWidgets() {
        for (TableItem tableItem : tableViewer.getTable().getItems()) {
            expression.setModel((IValidatableElement) tableItem.getData());
            if (expression.isValidatable()) {
                validateWidget();
            }
        }
    }

    private void validateWidget() {
        validator.setReturnType(expression.getReturnType());
        validator.validateExpression(expression.getText());

        boolean foundError = false;
        StringBuilder errorsStr = new StringBuilder();
        for (IProblem problem : validator.getProblems()) {
            if (problem.isError()) {
                if (foundError) {
                    errorsStr.append(System.getProperty("line.separator"));
                }
                errorsStr.append(" * ");
                errorsStr.append(problem.getMessage());
                foundError = true;
            }
        }

        // update model
        expression.getModel().setError(new Object[] { foundError, errorsStr.toString() });

        // triggers view update
        tableViewer.update(expression.getModel(),
                new String[] { (String) tableViewer.getColumnProperties()[columnIndex] });
    }

    public void setContextVariables(Collection<? extends Variable> variables) {
        this.contextVariables = variables;
    }

    public void setProject(IProject project) {
        this.project = project;
    }

    public void setReturnType(DataType returnType) {
        this.returnType = returnType;
    }

    public void setExpression(IExpression expressions) {
        this.expression = expressions;
    }

    public void reinitialize() {
        if (proposalAdapter != null) {
            proposalAdapter.setEnabled(false);
            initialize();
        }
    }

    public void addFocusListener(Control control) {
        control.addFocusListener(focusListener);
    }

    public void setupAndInitialize(List<WPVariable> variables, IProject project, IExpression expression) {
        this.contextVariables = Arrays.asList(variables.toArray(new Variable[0]));
        this.project = project;
        this.expression = expression;
        initialize();
    }

    private FocusListener focusListener = new FocusListener() {
        @Override
        public void focusLost(FocusEvent e) {
            IValidatableElement model = (IValidatableElement) ((StructuredSelection)tableViewer.getSelection()).getFirstElement();
            expression.setModel(model);

            if (expression.isValidatable()) {
                validateWidget();
            }
        }

        @Override
        public void focusGained(FocusEvent e) {
            IValidatableElement model = (IValidatableElement) ((StructuredSelection)tableViewer.getSelection()).getFirstElement();
            expression.setModel(model);

            if (expression.isValidatable()) {
                proposalAdapter.setEnabled(true);
                validateWidget();
            } else {
                proposalAdapter.setEnabled(false);
            }
        }
    };
}