/*
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.ui.editor.resourcetype;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.part.EditorPart;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.wst.sse.ui.StructuredTextEditor;

import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.formpage.XPathMappingPage;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

/**
 * The Resource Type Editor, which has multiple pages allowing editing the mapping files for
 * a Resource Type.<br>
 * This Editor allows edit the variables mapping and the xpath mapping for a given message
 * 
 * @author Patrik Araujo
 */
public class ResourceTypeMappingEditor extends FormEditor
{
	private static final String MESSAGE_TYPE_INCOMING = "incoming";
	private static final String MESSAGE_TYPE_OUTGOING = "outgoing";
	
	private IResourceTypeService rtService;

	/* Flag to indicate if this editor is dirty or not */
	private boolean editorModified = false;
	
	private StructuredTextEditor sourcePage;
	private XPathMappingPage xpathPage;

	/**
	 * Creates the multi-page ResourceType Editor.
	 */
	public ResourceTypeMappingEditor()
	{
		super();
	}
	
	public Object getResponseInput(){
		IEditorInput editorInput = getEditorInput();
		IFile responseFile = null;
		if(editorInput instanceof FileEditorInput){
			IPath filePath = ((FileEditorInput) editorInput).getFile().getFullPath();
			IProject project = ((FileEditorInput) editorInput).getFile().getProject();
			String messageFolderPath = filePath.removeLastSegments(2).removeFirstSegments(1).toString();
			if(project.exists()){
				IFolder messageFolder = project.getFolder(messageFolderPath);
				responseFile = messageFolder.getFile("mapping/incoming.xml");
			}
		}
		return responseFile;
	}
	
	private String getEditorTitle(){
		String title = null;
		this.rtService = ServiceFactory.createFactory().createResourceTypeService();
		IProject project = ((FileEditorInput) getEditorInput()).getFile().getProject();
		ResourceType rt = this.rtService.getResourceTypeByName(project.getName());
		String messageUid = getMessageUid();
		for (NodeDefinitionDTO nodeDefinitionDTO : rt.getNodes()) {
			try {
				NodeDefinition nodeDef = SemanticElementUtil.createNodeDefinition(nodeDefinitionDTO.getXml());
				if(nodeDef.getUid().equals(messageUid)){
					title = nodeDef.getName();
					break;
				}
			} catch (Exception e) {
			}
		}
		
		return title + "(" + messageUid + ")";
	}
	
	private String getMessageType(){
		String messageType = null;
		IProject project = ((FileEditorInput) getEditorInput()).getFile().getProject();
		ResourceType rt = this.rtService.getResourceTypeByName(project.getName());
		String messageUid = getMessageUid();
		if(rt.getMessage(messageUid).isIncoming()){
			messageType = MESSAGE_TYPE_INCOMING;
		}else{
			messageType = MESSAGE_TYPE_OUTGOING;
		}
		return messageType;
	}
	
	private String getMessageUid(){
		IPath filePath = ((FileEditorInput) getEditorInput()).getFile().getFullPath();
		String messageUid = filePath.removeLastSegments(2).removeFirstSegments(3).toString();
		return messageUid;
	}
	
	
	// ========================================================================
	// ===================== OVERRIDDEN METHODS
	// ========================================================================
	
	@Override
	public IEditorPart getActiveEditor() {
		/* Biggest hack I ever done in my life ;-) */
		return null;
	}
	
	@Override
	protected void addPages() {
		this.updateTitle();
		try {
			createEditorPages();
			setPagesText();
		} catch (PartInitException e) {
			Utility.reportError("Error initializing editor pages.", e);
		}
		
		setActivePage(0);
	}
	
	private void setPagesText() throws PartInitException{
		if(getMessageType().equals(MESSAGE_TYPE_INCOMING)){
			addPage(0, xpathPage, getEditorInput());
			addPage(1, sourcePage, getEditorInput());
		}else{
			addPage(0, sourcePage, getEditorInput());
			addPage(1, xpathPage, getEditorInput());
		}
		setPageText(0, "Request");
		setPageText(1, "Response");
	}
	
	private void createEditorPages(){
		sourcePage = new StructuredTextEditor();
		xpathPage = new XPathMappingPage(this);
	}
	
	/**
	 * Saves the multi-page editor's document.
	 */
	@Override
	public void doSave(IProgressMonitor monitor){
		int page = this.getActivePage();
		if(getMessageType().equals(MESSAGE_TYPE_INCOMING)){
			handleSaveAction(xpathPage, sourcePage, page, monitor);
		}else{
			handleSaveAction(sourcePage, xpathPage, page, monitor);
		}
		this.editorModified = false;
		editorDirtyStateChanged();
	}
	
	private void handleSaveAction(EditorPart first, EditorPart second, int page, IProgressMonitor monitor){
		switch (page) {
		case 0:
			first.doSave(monitor);
			break;
		case 1:
			second.doSave(monitor);
			break;
		}
	}
	
	/**
	 * @see org.eclipse.ui.part.EditorPart#doSaveAs()
	 */
	@Override
	public void doSaveAs()
	{
		return;
	}

	/**
	 * The Editor requires input which is an instance of <code>ResourceTypeEditorInput</code>.
	 * 
	 * @see org.eclipse.ui.forms.editor.FormEditor#init(org.eclipse.ui.IEditorSite,
	 *      org.eclipse.ui.IEditorInput)
	 */
	@Override
	public void init(IEditorSite site, IEditorInput editorInput) throws PartInitException
	{
		super.init(site, editorInput);
		Utility.setWindow(site.getWorkbenchWindow());
	}

	/**
	 * Cannot 'Save As' for a Resource Type
	 * 
	 * @see org.eclipse.ui.part.EditorPart#isSaveAsAllowed()
	 */
	public boolean isSaveAsAllowed()
	{
		return false;
	}

	/**
	 * @see org.eclipse.ui.forms.editor.FormEditor#isDirty()
	 */
	@Override
	public boolean isDirty()
	{
		return (this.editorModified || super.isDirty() || this.xpathPage.isDirty());
	}
	
	@Override
	public void editorDirtyStateChanged()
	{
		/* The Editor is now dirty or clean. If Dirty, update the editorModified flag */
		if ((!this.editorModified) && this.isDirty())
			this.editorModified = true;

		super.editorDirtyStateChanged();
	}

	/*
	 * Update the Title of the Editor
	 */
	private void updateTitle()
	{
		IEditorInput input = this.getEditorInput();
		this.setPartName(getEditorTitle());
		this.setTitleToolTip(input.getToolTipText());
	}
}
