package com.tandbergtv.watchpoint.studio.ui.model;

import org.jbpm.gd.jpdl.model.Action;
import org.jbpm.gd.jpdl.model.ActionElement;
import org.jbpm.gd.jpdl.model.CancelTimer;
import org.jbpm.gd.jpdl.model.CreateTimer;
import org.jbpm.gd.jpdl.model.Event;
import org.jbpm.gd.jpdl.model.Script;
import org.jbpm.gd.jpdl.model.TaskNode;

/**
 * Model representation of an Abstract TaskNode, which is the parent of Manual and Automatic
 * TaskNodes.
 * 
 * @author Imran Naqvi
 * 
 */
public class AbstractTaskNode extends TaskNode implements IResourceGroupElement, IDueDateElement {

	private static final String NODE_ENTER_EVENT = "node-enter";

	private static final String NODE_LEAVE_EVENT = "node-leave";

	ResourceGroup resourceGroup;

	ActionElement actionElement;

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IResourceGroupElement#setResourceGroup(ResourceGroup)
	 */
	public void setResourceGroup(ResourceGroup newResourceGroup) {
		ResourceGroup oldResourceGroup = this.resourceGroup;
		this.resourceGroup = newResourceGroup;
		firePropertyChange("resourceGroupAdd", oldResourceGroup, newResourceGroup);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.model.IResourceGroupElement#getResourceGroup()
	 */
	public ResourceGroup getResourceGroup() {
		return this.resourceGroup;
	}

	/**
	 * Sets the script for this node
	 * 
	 * @param newActionElement
	 */
	public void setScript(Script newActionElement) {
		ActionElement oldActionElement = actionElement;
		actionElement = newActionElement;
		firePropertyChange("script", oldActionElement, newActionElement);
	}

	/**
	 * Gets the script for this node.
	 * 
	 * @return the script
	 */
	public Script getScript() {
		if (actionElement instanceof Script) {
			return (Script) actionElement;
		}
		return null;
	}

	/**
	 * @return the dueDate
	 */
	public String getDueDate() {
		CreateTimer createTimer = null;

		for (Event event : this.getEvents()) {
			if (event.getType().equals(NODE_ENTER_EVENT)) {
				for (ActionElement action : event.getActionElements())
					if (action instanceof CreateTimer)
						createTimer = (CreateTimer) action;
			}
		}
		if (createTimer == null)
			return null;

		return createTimer.getDueDate();
	}

	/**
	 * @param newDueDate
	 *            the dueDate to set
	 */
	public void setDueDate(String newDueDate) {
		if (newDueDate == null) {
			removeTimerEvents();
			return;
		}
		CreateTimer createTimer = getNodeEnterCreateTimer();
		if (createTimer == null)
			return;
		createTimer.setDueDate(newDueDate);
	}

	/**
	 * Sets the action.
	 * 
	 * @param newActionElement
	 *            Action to set.
	 */
	public void setAction(Action newActionElement) {
		ActionElement oldActionElement = actionElement;
		actionElement = newActionElement;
		firePropertyChange("action", oldActionElement, newActionElement);
	}

	/**
	 * Gets the action.
	 * 
	 * @return the action
	 */
	public Action getAction() {
		if (actionElement instanceof Action) {
			return (Action) actionElement;
		}
		return null;
	}

	/**
	 * Sets the create timer.
	 * 
	 * @param newActionElement
	 *            the create timer to set
	 */
	public void setCreateTimer(CreateTimer newActionElement) {
		ActionElement oldActionElement = actionElement;
		actionElement = newActionElement;
		firePropertyChange("createTimer", oldActionElement, newActionElement);
	}

	/**
	 * Gets the create timer.
	 * 
	 * @return the create timer
	 */
	public CreateTimer getCreateTimer() {
		if (actionElement instanceof CreateTimer) {
			return (CreateTimer) actionElement;
		}
		return null;
	}

	/**
	 * Sets the cancel timer.
	 * 
	 * @param newActionElement
	 *            the cancel timer to set
	 */
	public void setCancelTimer(CancelTimer newActionElement) {
		ActionElement oldActionElement = actionElement;
		actionElement = newActionElement;
		firePropertyChange("cancelTimer", oldActionElement, newActionElement);
	}

	/**
	 * Gets the cancel timer.
	 * 
	 * @return the cancel timer
	 */
	public CancelTimer getCancelTimer() {
		if (actionElement instanceof CancelTimer) {
			return (CancelTimer) actionElement;
		}
		return null;
	}

	/*
	 * Removes Create timer and delete timer actions from node-enter and node-leave events
	 * respectively.
	 */
	private void removeTimerEvents() {
		for (Event event : this.getEvents()) {
			/* Remove Create Timer Action */
			if (event.getType().equals(NODE_ENTER_EVENT)) {
				for (ActionElement action : event.getActionElements())
					if (action instanceof CreateTimer)
						event.removeActionElement(action);
				if (event.getActionElements().length == 0)
					this.removeEvent(event);
			}
			/* Remove timer delete action */
			if (event.getType().equals(NODE_LEAVE_EVENT)) {
				for (ActionElement action : event.getActionElements()) {
					if ((action instanceof Action)
							&& ((Action) action).getRefName() != null &&
							((Action) action).getRefName().equals(
									TemplateActionRefNames.TIMER_DELETE_ACTION))
						event.removeActionElement(action);
				}
				if (event.getActionElements().length == 0)
					this.removeEvent(event);
			}
		}
	}

	/*
	 * Gets the create timer action specified in this node's node-enter event. If none exists, then
	 * a new create timer action is created in this nodes node-enter event. Also, if a node-enter
	 * event does not exist for this node, it is created.
	 * 
	 */
	private CreateTimer getNodeEnterCreateTimer() {
		Event nodeEnterEvent = AddTimerActionToEvent(NODE_ENTER_EVENT,
				TemplateActionRefNames.TIMER_FIRE_ACTION);

		/* Add timer delete action */
		AddTimerActionToEvent(NODE_LEAVE_EVENT, TemplateActionRefNames.TIMER_DELETE_ACTION);

		for (ActionElement action : nodeEnterEvent.getActionElements())
			if (action instanceof CreateTimer)
				return (CreateTimer) action;
		return null;
	}

	/*
	 * 
	 */
	private Event AddTimerActionToEvent(String eventName, String timerAction) {
		boolean newEvent = false;
		Event nodeEvent = null;
		/* Find node enter event */
		for (Event event : this.getEvents()) {
			if (!event.getType().equals(eventName))
				continue;
			nodeEvent = event;
		}

		/* Create node enter event if it does not exist */
		if (nodeEvent == null) {
			nodeEvent = (Event) getFactory().createById(
					SemanticElementConstants.EVENT_SEID);
			/* Add node enter event */
			this.addEvent(nodeEvent);
			nodeEvent.setType(eventName);
			newEvent = true;
		}
		Action action = null;

		if (eventName.equals(NODE_ENTER_EVENT)) {
			CreateTimer createTimer = null;
			if (!newEvent) {
				/* Find create timer action */
				for (ActionElement actionElement : nodeEvent.getActionElements()) {
					if (!(actionElement instanceof CreateTimer))
						continue;
					createTimer = (CreateTimer) actionElement;
				}
			}
			/* Create create timer action if it does not exist */
			if (createTimer == null) {
				createTimer = (CreateTimer) getFactory().createById(
						SemanticElementConstants.CREATE_TIMER_SEID);
				/* Add create timer action */
				nodeEvent.addActionElement(createTimer);
			}
			action = createTimer.getAction();
			if (action == null) {
				action = (Action) getFactory().createById(
						SemanticElementConstants.ACTION_SEID);
				createTimer.setAction(action);
			}
		}
		else if (eventName.equals(NODE_LEAVE_EVENT)) {
			for (ActionElement actionElement : nodeEvent.getActionElements()) {
				if (actionElement instanceof Action
						&& ((Action) actionElement).getRefName() != null
						&& ((Action) actionElement).getRefName().equals(timerAction))
					action = (Action) actionElement;
			}
			if (action == null) {
				action = (Action) getFactory().createById(
						SemanticElementConstants.ACTION_SEID);
				nodeEvent.addActionElement(action);
			}
		}

		action.setRefName(timerAction);

		return nodeEvent;
	}

}
