package com.tandbergtv.watchpoint.studio.ui.model;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.jbpm.gd.common.model.AbstractSemanticElement;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.NodeElementContainer;

import com.tandbergtv.watchpoint.studio.ui.editor.NodeDefinitionEditor;
import com.tandbergtv.watchpoint.studio.ui.editor.WatchPointTemplateEditor;

/**
 * Class that represents a loop node semantic element. <br>
 * A Loop Node is an node element that contains node elements. <br>
 * The Loop Node supports semantics such as Java "for" loop, Java "while" loop 
 * and Java "foreach" over a list.  
 */
public class LoopNode extends WPAbstractNode implements NodeElementContainer{
	private String index;
	private int initialValue;
	private String expression;
	private String listName;
	private List<NodeElement> nodes = new ArrayList<NodeElement>();	
	private LoopNodeType loopType = LoopNodeType.FOR;
	
	private static final String SPACE = " ";
	private static final String EMPTY_STRING = "";
	private static final String SINGLE_QUOTE = "'";
	private static final String EMPTY_OPERATOR = "empty ";

	@Override
	public void initialize() {
		this.setInitialValue(0);
	}

	@Override
	public void addNodeElement(NodeElement node) {
		nodes.add(node);
		firePropertyChange("nodeElementAdd", null, node);
		((WPNodeElement) node).setParent(this);
	}

	@Override
	public void removeNodeElement(NodeElement node) {
		nodes.remove(node);
		firePropertyChange("nodeElementRemove", node, null);
        if (node != null) {
            ((WPNodeElement) node).setParent(null);
        }
	}

	@Override
	public NodeElement[] getNodeElements() {
		return (NodeElement[]) nodes.toArray(new NodeElement[nodes.size()]);
	}

	@Override
	public NodeElement getNodeElementByName(String nodeName) {
		NodeElement[] nodeElements = getNodeElements();
		for (int i = 0; i < nodeElements.length; i++) {
			if (nodeElements[i].getName().equals(nodeName)) {
				return nodeElements[i];
			}
		}
		return null;
	}

	@Override
	public boolean canAdd(NodeElement node) {
		boolean canAdd = true;
		IWorkbenchWindow dwindow = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		IWorkbenchPage page = dwindow.getActivePage();
		IEditorPart activeEditor = page.getActiveEditor();
		/* Checks which editor is active and apply rules appropriately */
		if( activeEditor instanceof NodeDefinitionEditor ){
			canAdd = canAddForSuperstateEditor(node);
		} else if( activeEditor instanceof WatchPointTemplateEditor ){
			canAdd = canAddForTemplateEditor(node);
		}
		
		return canAdd;
	}
	
	public boolean canAddForTemplateEditor(NodeElement node){
		List<String> notAllowedElements = new ArrayList<String>();
		notAllowedElements.add(SemanticElementConstants.LOOPNODE_SEID);
		notAllowedElements.add(SemanticElementConstants.STARTSTATE_SEID);
		notAllowedElements.add(SemanticElementConstants.ENDSTATE_SEID);
		
		return !notAllowedElements.contains(node.getElementId());
	}
	
	public boolean canAddForSuperstateEditor(NodeElement node){
		List<String> notAllowedElements = new ArrayList<String>();
		notAllowedElements.add(SemanticElementConstants.LOOPNODE_SEID);
		notAllowedElements.add(SemanticElementConstants.ENDSTATE_SEID);
		
		return !notAllowedElements.contains(node.getElementId());
	}
	
	/* Set's and getter's -------- */
	
	public void setIndex(String index) {
		String oldIndex = this.index;
		this.index = index;
		firePropertyChange("index", oldIndex, index);
	}

	public String getIndex() {
		return index;
	}

	public void setInitialValue(int initialValue) {
		int oldInitialValue = this.initialValue;
		this.initialValue = initialValue;
		firePropertyChange("initialValue", oldInitialValue, Integer.toString(initialValue));
	}

	public int getInitialValue() {
		return initialValue;
	}

	public void setExpression(String expression) {
		String oldExpression = this.expression;
		this.expression = expression;
		firePropertyChange("expression", oldExpression, expression);
	}

	public String getExpression() {
		return expression;
	}

	public void setListName(String listName) {
		String oldListName = this.listName;
		this.listName = listName;
		firePropertyChange("list", oldListName, listName);
	}

	public String getListName() {
		return listName;
	}
	
	public int getFinalValue(){
		int finalValue = -1;
		if(expression != null){
			String[] operands = getLogicalOperands(expression);
			if(operands.length > 1){
				String operand = operands[1];
				if(isConstant(operand)){
					finalValue = Integer.parseInt(operand);
				}
			}
		}
		
		return finalValue;
	}
	
	/*
	 * Checks whether the operand is constant or not
	 */
	private static boolean isConstant(String operand) {
		try {
			new BigDecimal(operand);
			return true;
		} catch (NumberFormatException e) {
		}
		return false;
	}
	
	/*
	 * Gets the operands in the expression
	 */
	private String[] getLogicalOperands(String expression) {
		char[] operators = { '=', '|', '!', '&', '(', ')', '>', '<', '[', ']' };
		int index = 0;
		int exprLength = expression.length();
		String tempValue = EMPTY_STRING;
		StringBuffer operandValue = new StringBuffer();
		List<String> operands = new ArrayList<String>();

		while (index < exprLength) {
			char c = expression.charAt(index);
			boolean isValue = true;

			for (int j = 0; j < operators.length; j++) {
				if (c == operators[j]) {
					isValue = false;
					break;
				}
			}

			// Skip past other operators
			if (expression.startsWith(EMPTY_OPERATOR, index)) {
				index += EMPTY_OPERATOR.length();
				continue;
			}

			if (isValue) {
				operandValue.append(expression.charAt(index));
				tempValue = operandValue.toString().replaceAll(SPACE, EMPTY_STRING);
			}

			if ((!tempValue.equals(EMPTY_STRING) && !isValue) || (index == exprLength - 1 && isValue)) {
				if (tempValue.indexOf(SINGLE_QUOTE) == -1 )
					operands.add(operandValue.toString().trim());

				operandValue.delete(0, operandValue.length());
				tempValue = EMPTY_STRING;
			}

			index++;
		}
		if(operandValue.length() > 0){
			operands.add(operandValue.toString().trim());
		}

		return operands.toArray(new String[operands.size()]);
	}

	public void setLoopType(LoopNodeType loopType) {
		LoopNodeType oldType = this.loopType;
		this.loopType = loopType;
		String value = loopType != null ? loopType.toString() : null;
		firePropertyChange("type", oldType, value);
	}

	public LoopNodeType getLoopType() {
		return loopType;
	}

    @Override
    public void accept(SemanticElementVisitor visitor) {
        for (NodeElement nodeElement : getNodeElements()) {
            ((AbstractSemanticElement)nodeElement).accept(visitor);
        }
        visitor.visit(this);
    }
}
