/**
 * SuperStateComparer.java
 * Created May 27, 2010
 */
package com.tandbergtv.watchpoint.studio.ui.sync.compare;

import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.ADD;
import static com.tandbergtv.watchpoint.studio.ui.sync.DiffKind.REMOVE;

import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;

import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITree;
import com.tandbergtv.watchpoint.studio.ui.model.tree.ITreeNode;
import com.tandbergtv.watchpoint.studio.ui.model.tree.provider.MutableTreeNode;
import com.tandbergtv.watchpoint.studio.ui.sync.IDiff;
import com.tandbergtv.watchpoint.studio.ui.sync.impl.Diff;

/**
 * Compares two superstate node definitions
 * 
 * @author Sahil Verma
 */
public class SuperStateComparer extends NodeDefinitionComparer {

	int localElementsSize;
	int remoteElementsSize;
	int localTransitionsSize;
	int remoteTransitionsSize;

	/**
	 * {@inheritDoc}
	 */
	@Override
	protected ITree<IDiff> doCompare(IProgressMonitor monitor) {
		ITree<IDiff> tree = createDiffTree();
		ITreeNode<IDiff> root = tree.getRoot();
		
		try {
			
			if (monitor != null) {
				monitor.beginTask("Calculating changes to variables", 3);
			}
			
			NodeGroup localGroup = (NodeGroup) localElement.getNode();
			NodeGroup remoteGroup = (NodeGroup) remoteElement.getNode();
			
			// Cache the size cause each call to getNodeElements() converts the list into an array
			localElementsSize = localGroup.getNodeElements().length;
			remoteElementsSize = remoteGroup.getNodeElements().length;
			localTransitionsSize = localGroup.getTransitions().length;
			remoteTransitionsSize = remoteGroup.getTransitions().length;
			
			for (ITreeNode<IDiff> treenode : getAddedTreeNodes())
				root.addChild(treenode);
			
			incrementMonitor(monitor);
			
			for (ITreeNode<IDiff> treenode : getRemovedTreeNodes())
				root.addChild(treenode);
			
			incrementMonitor(monitor);
			
			for (ITreeNode<IDiff> treenode : getChangedTreeNodes())
				root.addChild(treenode);
			
			incrementMonitor(monitor);
			
		} finally {
			if (monitor != null) {
				monitor.done();
			}
		}
		
		return tree;
	}

	protected List<ITreeNode<IDiff>> getAddedTreeNodes() {
		List<ITreeNode<IDiff>> treenodes = super.getAddedTreeNodes();
		
		/*TODO Improve SuperState comparison by using NodeDefinition versioning or something like that. 
		 * 	   Ideally it should iterate over all Nodes in group comparing one-by-one the graph structure
		 * but that would kill the Builder.
		 * So, for now do just simple comparison...
		 * */
		if (remoteElementsSize > localElementsSize || remoteTransitionsSize > localTransitionsSize) {
			treenodes.add(new MutableTreeNode<IDiff>(new Diff(remoteElement, ADD)));
		}
		
		return treenodes;
	}

	protected List<ITreeNode<IDiff>> getRemovedTreeNodes() {
		List<ITreeNode<IDiff>> treenodes = super.getRemovedTreeNodes();
		
		/*TODO Improve SuperState comparison by using NodeDefinition versioning or something like that. 
		 * 	   Ideally it should iterate over all Nodes in group comparing one-by-one the graph structure
		 * but that would kill the Builder.
		 * So, for now do just simple comparison...
		 * */
		if (remoteElementsSize < localElementsSize || remoteTransitionsSize < localTransitionsSize) {
			treenodes.add(new MutableTreeNode<IDiff>(new Diff(localElement, REMOVE)));
		}
		
		return treenodes;
	}
}
