package com.tandbergtv.watchpoint.studio.ui.view.resourcetype;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IBaseLabelProvider;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.ITreeSelection;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeSelection;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.FileEditorInput;

import com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionDTO;
import com.tandbergtv.watchpoint.studio.dto.NodeDefinitionType;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.service.IResourceTypeService;
import com.tandbergtv.watchpoint.studio.service.ServiceFactory;
import com.tandbergtv.watchpoint.studio.ui.editor.input.IInputChangedListener;
import com.tandbergtv.watchpoint.studio.ui.editor.input.InputChangedEvent;
import com.tandbergtv.watchpoint.studio.ui.editor.resourcetype.ResourceTypeEditorInput;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.util.NameValuePair;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer;
import com.tandbergtv.watchpoint.studio.ui.view.DefaultContentProvider;
import com.tandbergtv.watchpoint.studio.ui.view.ViewAction;
import com.tandbergtv.watchpoint.studio.ui.wizard.export.ResourceTypeExportWizard;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

/**
 * The Resource Type Explorer View
 * 
 * @author Vijay Silva
 * @author DReddy
 */
public class ResourceTypeExplorer extends AbstractTreeViewExplorer {
	private static final String RESOURCE_TYPE_IMAGE_PATH = "resource_type.png";

	private static final String INCOMING_MESSAGE_IMAGE_PATH = "message_incoming.png";
	
	private static final String OUTGOING_MESSAGE_IMAGE_PATH = "message_outgoing.png";

	private static final String MESSAGE_VARIABLE_IMAGE_PATH = "icon_node.png";

	private ServiceFactory serviceFactory;

	private IResourceTypeService service;

	private ResourceTypeContentProvider contentProvider = new ResourceTypeContentProvider();

	private Action exportAction;

	/**
	 * Class Constructor
	 */
	public ResourceTypeExplorer() {
		super();

		this.serviceFactory = ServiceFactory.createFactory();
		this.service = this.serviceFactory.createResourceTypeService();
	}

	/**
	 * Create the ToolBar for the Explorer
	 * 
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#createToolbar()
	 */
	@Override
	protected void createToolbar() {
		IToolBarManager tbManager = getViewSite().getActionBars().getToolBarManager();
		
		tbManager.add(editAction);
		tbManager.add(exportAction);
	}

	@Override
	public void setFocus() {
		super.setFocus();
		this.refresh();
	}

	/**
	 * Fill the Context Menu for the Explorer Selection
	 * 
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#fillContextMenu(org.eclipse.jface.action.IMenuManager)
	 */
	@Override
	protected void fillContextMenu(IMenuManager mgr) {
		// Get the current selection
		Object selectedObject = this.getSelectedObject();
		if ((selectedObject instanceof ResourceType)) {
			mgr.add(this.editAction);
			mgr.add(this.exportAction);
		}
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#createActions()
	 */
	@Override
	protected void createActions() {
		super.createActions();

		this.exportAction = new ViewAction(this, "Export", Utility.getImageDescriptor("export.png")) {
			/**
			 * @see org.eclipse.jface.action.Action#run()
			 */
			@Override
			public void performAction() {
				ResourceTypeExplorer.this.export();
			}
		};
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	@Override
	public void selectionChanged(SelectionChangedEvent event) {
		Object element = getSelectedObject();
		
		if (element == null)
			return;
		
		if (element instanceof ResourceType) {
			super.editAction.setEnabled(true);
			this.exportAction.setEnabled(true);
		} else {
			editAction.setEnabled(false);
			exportAction.setEnabled(false);
		}
	}

	// ========================================================================
	// ===================== TOOLBAR ACTIONS
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#create()
	 */
	@Override
	protected void create() {
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#view()
	 */
	@Override
	protected void view() {
		this.openSelectedResourceTypeInEditor(false);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#edit()
	 */
	@Override
	protected void edit() {
		Object element = ((TreeSelection) getSelection()).getFirstElement();
		
		if (element instanceof ResourceType)
			this.openSelectedResourceTypeInEditor(true);
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#delete()
	 */
	@Override
	protected void delete() {
	}

	/**
	 * Export the Selected Resource Type
	 */
	protected void export() {
		ResourceType resourceType = this.getSelectedResourceType();
		long resourceTypeId = (resourceType != null) ? resourceType.getId()
				: -1L;
		if (resourceTypeId == -1)
			return;

		ResourceTypeExportWizard exportWizard = new ResourceTypeExportWizard();
		exportWizard.init(super.getViewSite().getWorkbenchWindow().getWorkbench(), super.getSelection());
		WizardDialog dialog = new WizardDialog(super.getViewSite().getShell(), exportWizard);
		dialog.open();
	}

	/*
	 * Opens the Editor showing the Resource Type selected.
	 */
	private void openSelectedResourceTypeInEditor(boolean editable) {
		ResourceType resourceType = this.getSelectedResourceType();
		long resourceTypeId = (resourceType != null) ? resourceType.getId()
				: -1L;

		/* No Resource Type selected */
		if (resourceTypeId == -1)
			return;

		this.showBusy(true);
		try {
			resourceType = this.getLatestResourceType(resourceTypeId);
			if (resourceType == null)
				return;

			if (editable && resourceType.isOutOfTheBox()) {
				MessageDialog.openInformation(super.getSite().getShell(),
						"Error",
						"The Resource Type is out-of-the-box, cannot edit.");
				return;
			}

			Message message = this.getSelectedMessage();
			ResourceTypeEditorInput input = null;
			input = new ResourceTypeEditorInput(resourceType, message, editable);
			input.addInputChangedListener(new ResourceTypeEditorInputChangedListener());
			this.showResourceTypeEditor(input);
		} finally {
			this.showBusy(false);
		}
	}

	// ========================================================================
	// ===================== EDITOR MANAGEMENT
	// ========================================================================

	/*
	 * Show the Editor
	 */
	private IEditorPart showResourceTypeEditor(IEditorInput input) {
		IEditorPart editorPart = null;
		ResourceType resourceType = (ResourceType)input.getAdapter(ResourceType.class);
		IFile pluginFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(resourceType.getPath()));
		if(pluginFile.exists()){
			FileEditorInput fileInput = new FileEditorInput(pluginFile);
			try {
				IWorkbenchPage workbenchPage = this.getSite().getPage();
				editorPart = IDE.openEditor(workbenchPage, fileInput,
						Utility.RESOURCE_TYPE_EDITOR_ID, true);
			} catch (PartInitException ex) {
				this.showBusy(false);
				Utility.reportError("Failed to open the Resource Type Editor.", ex);
			}
		}

		return editorPart;
	}

	// ========================================================================
	// ===================== BACKEND CALLS
	// ========================================================================

	/*
	 * Get the Resource Type from the database given the Resource Type id
	 */
	private ResourceType getLatestResourceType(long resourceTypeId) {
		ResourceType resourceType = null;

		try {
			resourceType = this.service.getResourceType(resourceTypeId);
		} catch (Exception ex) {
			this.showBusy(false);
			Utility.reportError("Failed to get the Resource Type.", ex);
		}

		return resourceType;
	}

	/*
	 * Get the Resource Type selected in the Viewer. If a Message is selected,
	 * returns the resource type contained in the message.
	 */
	private ResourceType getSelectedResourceType() {
		ResourceType selectedResourceType = null;

		Object selectedObject = getSelectedObject();
		if (selectedObject instanceof ResourceType) {
			selectedResourceType = (ResourceType) selectedObject;
		} else if (selectedObject instanceof Message) {
			Message message = (Message) selectedObject;
			selectedResourceType = message.getResourceType();
		}

		return selectedResourceType;
	}

	/*
	 * Get the Message selected in the Viewer. If a Resource Type is selected,
	 * return null.
	 */
	private Message getSelectedMessage() {
		Message selectedMessage = null;

		Object selectedObject = getSelectedObject();
		if (selectedObject instanceof Message) {
			selectedMessage = (Message) selectedObject;
		}

		return selectedMessage;
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/*
	 * Get the Object selected in the Tree Viewer
	 */
	private Object getSelectedObject() {
		Object selectedObject = null;

		ITreeSelection selection = this.getSelection();
		if (selection != null) {
			selectedObject = selection.getFirstElement();
		}

		return selectedObject;
	}

	// ========================================================================
	// ===================== TREE VIEWER CONTENT MANAGEMENT
	// ========================================================================

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#getContentProvider()
	 */
	@Override
	protected IContentProvider getContentProvider() {
		return this.contentProvider;
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#getLabelProvider()
	 */
	@Override
	protected IBaseLabelProvider getLabelProvider() {
		return new ResourceTypeLabelProvider();
	}

	/**
	 * @see com.tandbergtv.watchpoint.studio.ui.view.AbstractTreeViewExplorer#getInput()
	 */
	@Override
	protected List<? extends IWatchPointDTO> getInput() {
		List<ResourceType> resourceTypes = null;
		try {
			resourceTypes = service.getResourceTypeList();
		} catch (Exception ex) {
			Utility
					.reportError("Failed to get the list of Resource Types.",
							ex);
		}

		return resourceTypes;
	}

	/*
	 * The Label Provider for the Explorer
	 */
	private class ResourceTypeLabelProvider extends LabelProvider {
		/**
		 * @see org.eclipse.jface.viewers.LabelProvider#getText(java.lang.Object)
		 */
		@Override
		public String getText(Object element) {
			String text = null;
			if (element instanceof ResourceType) {
				ResourceType resourceType = (ResourceType) element;
				text = resourceType.getName() + " ["
						+ resourceType.getSystemId() + "]";
			}
			else if (element instanceof NodeDefinition) {
				NodeDefinition message = (NodeDefinition) element;
				text = message.getName();
				if (message.getNodeElements() != null
						&& message.getNodeType() == NodeDefinitionType.SuperState)
					text += " (SuperState)";
				else
					text += " (Message)";
			} else if (element instanceof WPVariable) {
				WPVariable var = (WPVariable) element;
				text = var.getMappedName();
				String readorwrite = (var.getAccess() != null)? var.getAccess():"";
				if (readorwrite.equals("write"))
					text += " : in";
				else if (readorwrite.equals("read"))
					text += " : out";
			} else if(element instanceof NameValuePair){
				NameValuePair var = (NameValuePair) element;
				text = var.getName() + " : out (Constant)";
			} else{
				text = super.getText(element);
			}

			return text;
		}

		/**
		 * @see org.eclipse.jface.viewers.LabelProvider#getImage(java.lang.Object)
		 */
		@Override
		public Image getImage(Object element) {
			// TODO Cache the images or share them. Not tested with images
			Image image = null;
			if (element instanceof ResourceType) {
				ResourceType resourceType = (ResourceType)element;
				if(hasErrors(resourceType.getName())){
					image = this.createImage("resource_type_error.png");
				}else{
					image = this.createImage(RESOURCE_TYPE_IMAGE_PATH);
				}
			} else if (element instanceof NodeDefinition) {
				NodeDefinition node = (NodeDefinition) element;
				String imagePath = OUTGOING_MESSAGE_IMAGE_PATH;
				if (node.isIncoming())
					imagePath = INCOMING_MESSAGE_IMAGE_PATH;

				image = this.createImage(imagePath);
			} else if (element instanceof WPVariable) {
				image = this.createImage(MESSAGE_VARIABLE_IMAGE_PATH);
			} else if (element instanceof NameValuePair){
				image = this.createImage(MESSAGE_VARIABLE_IMAGE_PATH);
			} else {
				image = super.getImage(element);
			}

			return image;
		}

		/*
		 * Create a new Image from the relative path, or null if the path is
		 * null.
		 */
		private Image createImage(String path) {
			if (path == null)
				return null;

			ImageDescriptor descriptor = Utility.getImageDescriptor(path);
			return new Image(Display.getCurrent(), descriptor.getImageData());
		}
		
		private boolean hasErrors(String resourceTypeName){
			try {
				IFile pluginFile = ResourcesPlugin.getWorkspace().getRoot().getProject(resourceTypeName).getFile("plugin.xml");
				if(pluginFile.exists()){
					IMarker[] markers = pluginFile.findMarkers("org.jbpm.gd.jpdl.resourceTypeProblem", false, IResource.DEPTH_ZERO);
					if(markers.length > 0){
						return true;
					}
				}
			} catch (CoreException e) {
				e.printStackTrace();
			}
			return false;
		}
	}

	// ========================================================================
	// ===================== THE CONTENT PROVIDER
	// ========================================================================

	/*
	 * The Resource Type Content Provider
	 */
	private class ResourceTypeContentProvider extends DefaultContentProvider {
		/**
		 * @see org.eclipse.jface.viewers.ArrayContentProvider#getElements(java.lang.Object)
		 */
		@SuppressWarnings("unchecked")
		@Override
		public Object[] getElements(Object inputElement) {
			List<ResourceType> resourceTypeList = (List<ResourceType>) inputElement;
			return super.getElements(resourceTypeList);
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object element) {
			Object[] result = null;

			if (element instanceof ResourceType) {
				ResourceType resourceType = (ResourceType) element;
				Set<NodeDefinitionDTO> nodes = resourceType.getNodes();
				Set<NodeDefinition> nodeDefs = null;
				
				if (nodes != null) {
					try {
						nodeDefs = new HashSet<NodeDefinition>();
						
						for (NodeDefinitionDTO node : nodes) {
							NodeDefinition nodeDef = SemanticElementUtil.createNodeDefinition(node.getXml());
							nodeDef.setId(node.getId());
							nodeDefs.add(nodeDef);
							
							for (Message message : resourceType.getMessages()) {
								if (message.getName().equals(node.getName())) {
									nodeDef.setIncoming(message.isIncoming());
								}
							}
						}
						result = nodeDefs.toArray();
					} catch (Exception e) {
					}
				}
			} else if (element instanceof NodeDefinition) {
				NodeDefinition node = (NodeDefinition) element;
				
				List<Object> vars = new ArrayList<Object>();
				vars.addAll(node.getVariables());
				vars.addAll(node.getConstants());
				
				if (vars != null && vars.size() > 0)
					result = vars.toArray();
			} else {
				result = super.getChildren(element);
			}

			return result;
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
		 */
		public Object getParent(Object element) {
			Object parent = null;

			if (element instanceof NodeDefinition) {
				NodeDefinition message = (NodeDefinition) element;
				parent = service.getResourceType(message.getResourceType());
			} else if (element instanceof WPVariable) {
			} else {
				parent = super.getParent(element);
			}

			return parent;
		}

		/**
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
		 */
		public boolean hasChildren(Object element) {
			boolean result = false;

			if (element instanceof ResourceType) {
				ResourceType resourceType = (ResourceType) element;
				Set<NodeDefinitionDTO> messages = resourceType.getNodes();
				result = (messages != null && messages.size() > 0);
			} else if (element instanceof NodeDefinition) {
				NodeDefinition node = (NodeDefinition) element;
				List<Object> vars = new ArrayList<Object>();
				vars.addAll(node.getVariables());
				vars.addAll(node.getConstants());
				result = (vars != null && vars.size() > 0);
			} else {
				result = super.hasChildren(element);
			}

			return result;
		}
	}

	// ========================================================================
	// ===================== THE INPUT CHANGE LISTENER
	// ========================================================================

	/*
	 * The Label Provider for the Explorer
	 */
	private class ResourceTypeEditorInputChangedListener implements
			IInputChangedListener {
		/**
		 * @see com.tandbergtv.watchpoint.studio.ui.editor.input.IInputChangedListener#inputChanged(com.tandbergtv.watchpoint.studio.ui.editor.input.InputChangedEvent)
		 */
		public void inputChanged(InputChangedEvent event) {
		}
	}
	
}
