/*
 * Created on Nov 12, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.nodeelement;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.jbpm.gd.jpdl.model.NodeElement;
import org.jbpm.gd.jpdl.model.Variable;

import com.tandbergtv.watchpoint.studio.ui.editor.expression.ExpressionUtil;
import com.tandbergtv.watchpoint.studio.ui.model.NodeDefinition;
import com.tandbergtv.watchpoint.studio.ui.model.NodeGroup;
import com.tandbergtv.watchpoint.studio.ui.model.WPAbstractNode;
import com.tandbergtv.watchpoint.studio.ui.model.WPVariable;
import com.tandbergtv.watchpoint.studio.ui.properties.DataType;
import com.tandbergtv.watchpoint.studio.ui.util.Utility;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;
import com.tandbergtv.watchpoint.studio.validation.model.ContainerNode;
import com.tandbergtv.watchpoint.studio.validation.rules.expression.VariableExpressionValidator;

/**
 * Validation Rule that checks to see if every Variable is mapped to some Container Variable and if
 * the types of both variables match, if specified.
 * 
 * @author Vijay Silva
 */
public abstract class NodeVariablesMappingRule extends ValidationRule<ContainerNode> {

    /**
	 * @param target
	 *            The Container Node to validate
	 * @return The list of validation messages
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(ContainerNode target) {
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		List<WPVariable> containerVariables = target.getContainerVariables();
		List<WPVariable> nodeVariables = this.getVariables(target.getNode());
		Map<Variable, WPVariable> variableMapping = new HashMap<Variable, WPVariable>();

		if (nodeVariables != null) {
			for (WPVariable variable : nodeVariables) {
				WPVariable match = this.findContainerVariable(variable, containerVariables);
				
				variableMapping.put(variable, match);
			}
		}

		this.validateMissingMapping(target, variableMapping, messages);
		this.validateInvalidMapping(target, variableMapping, messages);
		this.validateMappingTypes(target, variableMapping, messages);

		return messages;
	}

	/**
	 * Get a list of the variables for the Node Element.
	 * 
	 * @param node
	 *            the Node Element.
	 * 
	 * @return the list of variables belonging to the node.
	 */
	protected abstract List<WPVariable> getVariables(NodeElement node);

	/**
	 * Validates if the any of the node variables are missing a mapping to one of the container
	 * variables.
	 * 
	 * @param node
	 *            the Container Node
	 * @param mapping
	 *            the mapping between the node variables and the container variables
	 * @param messages
	 *            the list of validation messages
	 */
	protected void validateMissingMapping(ContainerNode node, Map<Variable, WPVariable> mapping, List<ValidationMessage> messages) {
		NodeElement target = node.getNode();

		for (Variable variable : mapping.keySet()) {
			String name = variable.getName();

			if (isBlank(name)) {
				ValidationMessageCode code = getMappingMissingCode(node);
				List<String> parameters = new ArrayList<String>();
				parameters.add(variable.getMappedName());

                ValidationMessage validationMessage = ValidationMessageAdder.getInstance().addValidationMessage(messages, target, code, parameters);
				if (validationMessage != null) {
	                validationMessage.set(KEY_CONTAINER_VARIABLES, node.getContainerVariables());
	                validationMessage.set(KEY_MISSING_MAPPING_VARIABLE_NAME, variable.getMappedName());
	                validationMessage.set(KEY_MISSING_MAPPING_VARIABLE_TYPE, ((WPVariable)variable).getType());
				}
			}
		}
	}

	/**
	 * Validates that if a node variable is mapped, that the mapped variable is an existing
	 * Container Variable.
	 * 
	 * @param node
	 *            the Container Node
	 * @param mapping
	 *            the mapping between the node variables and the container variables
	 * @param messages
	 *            the list of validation messages
	 */
	protected void validateInvalidMapping(ContainerNode node,
			Map<Variable, WPVariable> mapping, List<ValidationMessage> messages) {
		NodeElement target = (NodeElement) node.getNode();

		VariableExpressionValidator variableExpressionValidator = null;
		for (Variable variable : mapping.keySet()) {
			String name = variable.getName();
			
			if (!this.isBlank(name) && mapping.get(variable) == null) {
				ValidationMessageCode code = this.getMappingInvalidCode(node);
				
				if (node.isTemplateContainer()) {
					boolean isWritable = variable.isWritable();
					
					if (target instanceof NodeDefinition && ((NodeDefinition)target).getNode() instanceof NodeGroup) {
						NodeDefinition nd = (NodeDefinition) target;
						NodeGroup group = (NodeGroup) nd.getNode();
						
						for (NodeElement ssElement : group.getNodeElements()) {
							if (ssElement instanceof NodeDefinition) {
								NodeDefinition ssNode = (NodeDefinition) ssElement;
								WPVariable ssVar = null; 
								
								for (WPVariable ssNodeVar : ssNode.getVariables()) {
									if (ssNodeVar.getName().equals(variable.getMappedName())) {
										ssVar = ssNodeVar;
										break;
									}
								}
								if (ssVar != null) {
									isWritable = ssVar.isWritable();
									break;
								}
							}
						}
					}
					if (!isWritable) {
						// Variable wasn't found in Template
						// and isn't writable. Tries to validate the script.
						if (variableExpressionValidator == null) {
				            String templateLocation = Utility.getTemplateLocation((WPAbstractNode) node.getNode());

                            variableExpressionValidator = new VariableExpressionValidator();
                            variableExpressionValidator.setProject(ResourcesPlugin.getWorkspace().getRoot()
                                    .getFile(new Path(templateLocation)).getProject());
                            variableExpressionValidator.setContextVariables(ExpressionUtil.toJavaVariableStr(node.getContainerVariables()));
						}
						
						WPVariable v = (WPVariable) variable;
						DataType t = DataType.STRING;
						
						if (v.getType() != null)
							t = DataType.valueOf(v.getType());
						
						variableExpressionValidator.setReturnType(t);
						variableExpressionValidator.validateExpression(variable.getName());
						if (!variableExpressionValidator.hasErrors()) {
							// Its a scripted value, not an error, continue.
							continue;
						} else {
							// Adds specific error.
							code = ValidationMessageCode.NODE_VARIABLE_MAPPING_INVALID_SCRIPT;
						}
					}
				}
				List<String> parameters = new ArrayList<String>();
				parameters.add(variable.getMappedName());
				parameters.add(variable.getName());
				ValidationMessageAdder.getInstance().addValidationMessage(messages, target, code, parameters);
			}
		}
	}
	
	/**
	 * Validates that the types of a variable in a node and the variable in the container to which
	 * the node variable is mapped to are the same when both types are present.
	 * 
	 * @param node
	 *            The Node being validated.
	 * @param mapping
	 *            The map containing the mapping between node variable and container variables.
	 * @param messages
	 *            The list of validation messages.
	 */
	protected void validateMappingTypes(ContainerNode node,
			Map<Variable, WPVariable> mapping, List<ValidationMessage> messages) {
		NodeElement target = node.getNode();
		for (Variable variable : mapping.keySet()) {
			WPVariable mappedVariable = mapping.get(variable);
			String type = (variable instanceof WPVariable) ? ((WPVariable) variable).getType() : null;
			String mappedType = (mappedVariable != null) ? mappedVariable.getType() : null;

			if ((!this.isBlank(type)) && (!this.isBlank(mappedType)) && (!type.equals(mappedType))) {
				ValidationMessageCode code = this.getMappingTypeMismatchCode(node);
				List<String> parameters = new ArrayList<String>();
				parameters.add(variable.getMappedName());
				parameters.add(mappedType);
				parameters.add(type);
				ValidationMessageAdder.getInstance().addValidationMessage(messages, target, code, parameters);
			}
		}
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/**
	 * Gets the Validation Message Code if the variable is not mapped to an existing container
	 * variable.
	 * 
	 * @param node
	 *            The node being validated.
	 * 
	 * @return {@link ValidationMessageCode#NODE_VARIABLE_NOT_MAPPED_IN_TEMPLATE} if the container
	 *         is a template, or
	 *         {@link ValidationMessageCode#NODE_VARIABLE_NOT_MAPPED_IN_NODE_DEFINITION} if the
	 *         container is a Node Definition.
	 */
	protected ValidationMessageCode getMappingMissingCode(ContainerNode node) {
		ValidationMessageCode code = null;

		if (node.isTemplateContainer()) {
			code = ValidationMessageCode.NODE_VARIABLE_NOT_MAPPED_IN_TEMPLATE;
		} else if (node.isNodeDefinitionContainer()) {
			code = ValidationMessageCode.NODE_VARIABLE_NOT_MAPPED_IN_NODE_DEFINITION;
		}

		return code;
	}

	/**
	 * Gets the Validation Message Code if the variable is mapped to a non-existing container
	 * variable.
	 * 
	 * @param node
	 *            The node being validated.
	 * 
	 * @return {@link ValidationMessageCode#NODE_VARIABLE_MAPPING_INVALID_IN_TEMPLATE} if the
	 *         container is a template, or
	 *         {@link ValidationMessageCode#NODE_VARIABLE_MAPPING_INVALID_IN_NODE_DEFINITION} if the
	 *         container is a Node Definition.
	 */
	protected ValidationMessageCode getMappingInvalidCode(ContainerNode node) {
		ValidationMessageCode code = null;

		if (node.isTemplateContainer())
			code = ValidationMessageCode.NODE_VARIABLE_MAPPING_INVALID_IN_TEMPLATE;
		else if (node.isNodeDefinitionContainer())
			code = ValidationMessageCode.NODE_VARIABLE_MAPPING_INVALID_IN_NODE_DEFINITION;

		return code;
	}

	/**
	 * Gets the Validation Message Code if the variable type mismatches with the container variable
	 * type.
	 * 
	 * @param node
	 *            The node being validated.
	 * 
	 * @return {@link ValidationMessageCode#NODE_VARIABLE_TYPE_MISMATCH_IN_TEMPLATE} if the
	 *         container is a template, or
	 *         {@link ValidationMessageCode#NODE_VARIABLE_TYPE_MISMATCH_IN_NODE_DEFINITION} if the
	 *         container is a Node Definition.
	 */
	protected ValidationMessageCode getMappingTypeMismatchCode(ContainerNode node) {
		ValidationMessageCode code = null;

		if (node.isTemplateContainer())
			code = ValidationMessageCode.NODE_VARIABLE_TYPE_MISMATCH_IN_TEMPLATE;
		else if (node.isNodeDefinitionContainer())
			code = ValidationMessageCode.NODE_VARIABLE_TYPE_MISMATCH_IN_NODE_DEFINITION;

		return code;
	}

	// ========================================================================
	// ===================== HELPER METHODS
	// ========================================================================

	/* Checks if a value is blank or null */
	private boolean isBlank(String value) {
		return (value == null || value.trim().length() == 0);
	}

	/* Find a Container variable with a matching name */
	private WPVariable findContainerVariable(WPVariable v, List<WPVariable> variables) {
		String name = v.getName();
		
		if (this.isBlank(name) || variables == null)
			return null;

		if (v.isListExpression())
			name = v.getListName();
		
		for (WPVariable variable : variables) {
			if (name.equals(variable.getName()))
				return variable;
		}

		return null;
	}
}
