/*
 * Created on Aug 28, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.resourcetype;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import com.tandbergtv.watchpoint.studio.dto.AdaptorType;
import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Rule that performs validation of the Resource Type properties related to adaptor configuration.
 * 
 * @author Vijay Silva
 */
public class AdaptorPropertiesRule extends ValidationRule<ResourceType>
{
	/**
	 * Validates the Adaptor properties of the Resource Type
	 * 
	 * @param target
	 *            The Resource Type to validate
	 * 
	 * @return The list of validation messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(ResourceType target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		this.validateAdaptorType(target, messages);
		this.validateCustomAdaptorProperties(target, messages);
		this.validateDefaultAdaptorProperties(target, messages);

		return messages;
	}

	/*
	 * Validate the Adaptor Type
	 */
	private void validateAdaptorType(ResourceType resourceType, List<ValidationMessage> messages)
	{
		AdaptorType adaptorType = resourceType.getAdaptorType();
		Set<Message> rtMessages = resourceType.getMessages();
		int messageCount = (rtMessages == null) ? 0 : rtMessages.size();

		if (adaptorType == null)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_ADAPTOR_TYPE_BLANK);
		}
		else if ((adaptorType == AdaptorType.NONE) && messageCount > 0)
		{
			ValidationMessageCode code = ValidationMessageCode.RESOURCE_TYPE_ADAPTOR_TYPE_INVALID;
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, code);
		}
	}

	/*
	 * Validate the Adaptor Properties if the type is custom adaptor type
	 */
	private void validateCustomAdaptorProperties(ResourceType resourceType,
			List<ValidationMessage> messages)
	{
		if (!(resourceType.getAdaptorType() == AdaptorType.CUSTOM))
			return;

		// Must have a Custom Adaptor Class Name defined
		String className = resourceType.getAdaptorClassName();
		if (className == null || className.trim().length() == 0)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_CUSTOM_ADAPTOR_CLASSNAME_BLANK);
		}
		else if (className.length() > 256)
		{
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_CUSTOM_ADAPTOR_CLASSNAME_TOO_LONG);
		}
	}

	/*
	 * Validate the Adaptor Properties if the type is default adaptor type
	 */
	private void validateDefaultAdaptorProperties(ResourceType resourceType,
			List<ValidationMessage> messages)
	{
		if (!(resourceType.getAdaptorType() == AdaptorType.DEFAULT))
			return;

		String messageIdentifier = resourceType.getMessageIdentificationClassName();
        if (messageIdentifier != null && messageIdentifier.length() > 256) {
            ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, ValidationMessageCode.RESOURCE_TYPE_MESSAGE_IDENTIFIER_TOO_LONG);
        }

	}

}
