/*
 * Created on Aug 28, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.validation.rules.resourcetype;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.tandbergtv.watchpoint.studio.dto.Message;
import com.tandbergtv.watchpoint.studio.dto.ResourceType;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessageCode;
import com.tandbergtv.watchpoint.studio.validation.ValidationRule;
import com.tandbergtv.watchpoint.studio.validation.impl.ValidationMessageAdder;

/**
 * Rule that validates each of the Messages of a Resource Type
 * 
 * @author Vijay Silva
 */
public class MessagesPropertyRule extends ValidationRule<ResourceType>
{
	/**
	 * Validates the Messages contained by the Resource Type
	 * 
	 * @param target
	 *            The ResourceType to validate
	 * 
	 * @return The list of Validation Messages
	 * 
	 * @see com.tandbergtv.watchpoint.studio.validation.IValidationRule#validateRule(java.lang.Object)
	 */
	public List<ValidationMessage> validateRule(ResourceType target)
	{
		List<ValidationMessage> messages = new ArrayList<ValidationMessage>();

		// Validate the Messages
		this.validateMessages(target, messages);

		return messages;
	}

	/*
	 * Validate the list of Messages
	 */
	private void validateMessages(ResourceType resourceType, List<ValidationMessage> messages)
	{
		Set<Message> rtMessages = resourceType.getMessages();

		if (rtMessages == null)
			return;

		// Validate for duplicate UIDs and Names
		Set<String> uids = new HashSet<String>();
		Set<String> duplicateUids = new HashSet<String>();
		Set<String> names = new HashSet<String>();
		Set<String> duplicateNames = new HashSet<String>();

		for (Message message : rtMessages)
		{
			/* Validate the Message Resource Type */
			ResourceType containedResourceType = message.getResourceType();
            if ((!resourceType.equals(containedResourceType)) && containedResourceType != null) {
                ValidationMessageAdder.getInstance().addValidationMessage(messages, message, ValidationMessageCode.RESOURCE_TYPE_MESSAGE_RESOURCE_TYPE_INVALID);
            }

			// Collect all duplicate names and UIDs
			boolean skipUid = false;
			boolean skipName = false;
			String uid = message.getUid();
			String name = message.getName();

			/* Skip blank UIDs or UIDs requiring generation */
			if (message.isUIDGenerationRequired() || uid == null || uid.trim().length() == 0)
				skipUid = true;

			/* Skip blank names or convert to trimmed lower case names only */
			if (name == null || name.trim().length() == 0)
				skipName = true;
			else
				name = name.trim().toLowerCase();

			/* Collect all the UIDs, making note of duplicate UIDs */
			if (!skipUid)
			{
				if (uids.contains(uid))
					duplicateUids.add(uid);
				else
					uids.add(uid);
			}

			if (!skipName)
			{
				if (names.contains(name))
					duplicateNames.add(message.getName());
				else
					names.add(name);
			}
		}

		/* Create Validation Messages for the Duplicate UIDs */
		this.validateDuplicateUIDs(resourceType, duplicateUids, messages);

		/* Create Validation Messages for the Duplicate Names */
		this.validateDuplicateNames(resourceType, duplicateNames, messages);
	}

	/*
	 * Create a Validation Error for the duplicate UIDs.
	 */
	private void validateDuplicateUIDs(ResourceType resourceType, Set<String> duplicateUids,
			List<ValidationMessage> messages)
	{
		for (String uid : duplicateUids)
		{
			ValidationMessageCode code = ValidationMessageCode.RESOURCE_TYPE_UID_DUPLICATES_ERROR;
			List<String> parameters = new ArrayList<String>();
			parameters.add(uid);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, code, parameters);
		}
	}

	/*
	 * Create a Validation Error for the duplicate Names.
	 */
	private void validateDuplicateNames(ResourceType resourceType, Set<String> duplicateNames,
			List<ValidationMessage> messages)
	{
		for (String name : duplicateNames)
		{
			ValidationMessageCode code = ValidationMessageCode.RESOURCE_TYPE_MESSAGE_NAME_DUPLICATES_ERROR;
			List<String> parameters = new ArrayList<String>();
			parameters.add(name);
			ValidationMessageAdder.getInstance().addValidationMessage(messages, resourceType, code, parameters);
		}
	}
}
