/*
 * Created on Aug 10, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.external.wpexport.impl;

import static com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes.EXPORT_FOLDER_PATH;
import static com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes.PLUGIN_CLASSES_FILE_PATHS;
import static com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes.PLUGIN_JAR_FILE_PATHS;
import static com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes.PLUGIN_RESOURCE_FILE_PATHS;
import static com.tandbergtv.watchpoint.studio.external.wpexport.impl.WatchPointPluginConstants.PLUGIN_LIB_FOLDER_NAME;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.jbpm.gd.common.xml.XmlAdapter;
import org.w3c.dom.Document;

import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.ui.model.WorkflowTemplate;
import com.tandbergtv.watchpoint.studio.util.FileUtil;
import com.tandbergtv.watchpoint.studio.util.SemanticElementUtil;

/**
 * Workflow Template exporter that exports the template to a file
 * 
 * @author Vijay Silva
 */
public class WorkflowTemplateFileExporter extends WorkflowTemplateExporter {
	/**
	 * @param template
	 *            The Workflow Template to export
	 * @param exportInputs
	 *            The input parameters for the export process. Input parameter
	 *            EXPORT_FILE_PATH must be present and have the target file path
	 *            as the value.
	 * 
	 * @return The output parameters from the export process. Returns an empty
	 *         map.
	 * 
	 * @throws ExportFailureException
	 *             Exception thrown in case the export process fails.
	 * 
	 * @see com.tandbergtv.watchpoint.studio.external.wpexport.ExporterInputAttributes#EXPORT_FILE_PATH
	 * @see com.tandbergtv.watchpoint.studio.external.wpexport.IWatchPointDTOExporter#export(com.tandbergtv.watchpoint.studio.dto.IWatchPointDTO,
	 *      java.util.Map)
	 */
	@Override
	public Map<String, Object> export(WorkflowTemplateDTO template, Map<String, Object> exportInputs)
			throws ExportFailureException {
		IFile sourceFile = ResourcesPlugin.getWorkspace().getRoot().getFile(new Path(template.getPath()));
		IFile destFile = sourceFile.getParent().getFile(new Path(WorkflowTemplateDTO.TEMPLATE_RUNTIME_FILE_NAME));

		try {
			String xml = new String(FileUtil.readFile(sourceFile.getContents()));

			/* Carriage return is stripped out for some reason... */
			xml = xml.replaceAll("\r", "");

			Map<String, Object> modelData = createWorkflowTemplate(xml);

			XmlAdapter adapter = SemanticElementUtil.getAdapter(modelData);

			WorkflowTemplate templateModel = (WorkflowTemplate) SemanticElementUtil.getSemanticElement(modelData);

			postProcess(templateModel);

			/* Get the JPDL and GPD */
			Document templateDocument = adapter.getNode().getOwnerDocument();
			
			templateDocument.setXmlStandalone(false);

			ByteArrayOutputStream out = new ByteArrayOutputStream();

			FileUtil.writeDocument(templateDocument, out);

			InputStream in = new ByteArrayInputStream(out.toByteArray());

			if (destFile.exists()) {
				destFile.setContents(in, IFile.FORCE, null);
			} else {
				destFile.create(in, IFile.FORCE, null);
			}
			return new HashMap<String, Object>();
		} catch (ExportFailureException efe) {
			throw efe;
		} catch (Exception ex) {
			String msg = "Failed to export the template";
			throw new ExportFailureException(msg, ex);
		}
	}

	/**
	 * Writes out JARs and resources to lib/
	 * 
	 * @param template
	 * @param properties
	 * @throws ExportFailureException
	 */
	@SuppressWarnings("unchecked")
	protected void writeFiles(WorkflowTemplateDTO template, Map<String, Object> properties)
			throws ExportFailureException {
		File libdir = new File((String) properties.get(EXPORT_FOLDER_PATH), PLUGIN_LIB_FOLDER_NAME);

		if (!libdir.exists()) {
			boolean created = libdir.mkdirs();

			if (!created)
				throw new ExportFailureException("Failed to create lib directory");
		}

		overwrite(libdir, (List<String>) properties.get(PLUGIN_JAR_FILE_PATHS));
		overwrite(libdir, (List<String>) properties.get(PLUGIN_CLASSES_FILE_PATHS));
		overwrite(libdir, (List<String>) properties.get(PLUGIN_RESOURCE_FILE_PATHS));
	}

	protected void overwrite(File dir, List<String> files) throws ExportFailureException {
		for (String file : files) {
			File source = new File(file);

			if (!source.exists())
				throw new ExportFailureException("File not found: " + file);

			File destination = new File(dir, source.getName());
			boolean copied = FileUtil.overwrite(source, destination, true);

			if (!copied)
				throw new ExportFailureException("Failed to copy " + file + " during template export");
		}
	}
}
