/*
 * Created on Jun 21, 2007
 * 
 * (C) Copyright TANDBERG Television Ltd.
 */

package com.tandbergtv.watchpoint.studio.service;

import java.util.List;

import com.tandbergtv.watchpoint.studio.dto.WorkflowTemplateDTO;
import com.tandbergtv.watchpoint.studio.external.wpexport.ExportFailureException;
import com.tandbergtv.watchpoint.studio.ui.editor.WatchPointTemplateEditor;
import com.tandbergtv.watchpoint.studio.validation.ValidationMessage;

/**
 * Service interface defining operations on WorkflowTemplateDTO entities.
 * 
 * @author Vijay Silva
 */
public interface IWorkflowTemplateService extends IService
{
	/**
	 * Get the list of latest (most recent version) active Templates.
	 * 
	 * @return The List of active Templates that have the more recent version number.
	 */
	List<WorkflowTemplateDTO> getTemplateList();
	
	/**
	 * Get the Templates for a specified project.
	 * 
	 * @return The List of templates for that project.
	 */
	List<WorkflowTemplateDTO> getTemplatesByProject(String projectName);
	
	/**
	 * Get the Templates which refers to a specified SuperState node.
	 * 
	 * @param nodeDefinitionName
	 * @param systemID
	 * @return
	 */
	List<WorkflowTemplateDTO> getTemplatesBySuperStateUsage(String nodeDefinitionName, String systemID);

	/**
	 * Creates a new Template.
	 * 
	 * @param template
	 *            The Template
	 * @param userId
	 *            The Id of the user creating the template
	 * @param lockTemplate
	 *            Flag to indicate if the newly created template needs to be locked
	 * 
	 * @return The created Workflow Template
	 */
	WorkflowTemplateDTO createTemplate(WorkflowTemplateDTO template, String userId,
			boolean lockTemplate);

	/**
	 * Get a Template. 
	 * 
	 * @param templateName
	 *            The template name.
	 * 
	 * @return The Template.
	 */
	List<WorkflowTemplateDTO> getTemplateByName(String templateName);
	
	/**
	 * Get all Templates that refers to a specified template. 
	 * 
	 * @param templateName
	 *            The template name.
	 * 
	 * @return The list of Template.
	 */
	List<WorkflowTemplateDTO> getReferencingTemplates(String templateName);
	
	/**
	 * Get all Templates that refers to a Resource Type. 
	 * 
	 * @param systemID
	 *            The Resource Type systemID.
	 * 
	 * @return The list of Template.
	 */
	List<WorkflowTemplateDTO> getTemplatesByResourceTypeUsage(String systemID);
	
	/**
	 * Get all Templates that refers to a ResourceGroup. 
	 * 
	 * @param resourceGroupName
	 *            The Resource Group name.
	 * 
	 * @return The list of Template.
	 */
	List<WorkflowTemplateDTO> getTemplatesByResourceGroupUsage(String resourceGroupName);
	
	/**
	 * Get a Template by its path. 
	 * 
	 * @param path
	 *            The Template path.
	 * 
	 * @return The Template.
	 */
	WorkflowTemplateDTO getTemplateByPath(String path);

	/**
	 * Saves the Template. The Template must be currently locked by the current user.
	 * 
	 * @param template
	 *            The Template to save.
	 * @param userId
	 *            The Id of the user saving the template.
	 * 
	 * @return The updated Template from the persistence layer.
	 */
	WorkflowTemplateDTO saveTemplate(WorkflowTemplateDTO template);
	
	/**
	 * Delete a Template by its path. 
	 * 
	 * @param templatePath
	 *            The Path for the Template to delete.
	 */
	void deleteTemplate(String templatePath);

	/**
	 *  	Export a template.
	 *  
	 * @param document
	 * @return
	 * @throws ExportFailureException 
	 */
	WorkflowTemplateDTO exportTemplate(WatchPointTemplateEditor documentFile) throws ExportFailureException;

	/**
	 * 		Validates a template. 
	 * 
	 * @param template
	 * @return
	 */
	List<ValidationMessage> validateTemplate(WorkflowTemplateDTO template);
}
